#include <sys/ipc.h>
#include <sys/shm.h>
#include <sys/time.h>
#include <math.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include "xvideoshow.hpp"



extern "C"
{
  void MyEventsOpen(Display *mydpy, Window w);
  int  MyEventsTrace(unsigned long *value);
  void MyEventsClose();
}



XVideoShow::XVideoShow(char *Name, int ImageWidth, int ImageHeight)   
{
  m_WindowName    = Name;
  m_Width         = ImageWidth;
  m_Height        = ImageHeight;
  m_Display       = 0;
  m_Image         = 0;
  m_VideoPort     = 0;

  memset(&m_VisInfo,0,sizeof(XVisualInfo));
  memset(&m_Xswa,   0,sizeof(XSetWindowAttributes));
  memset(&m_Window, 0,sizeof(Window));
  memset(&m_ShmInfo,0,sizeof(XShmSegmentInfo));
}



XVideoShow::~XVideoShow()
{
  MyEventsClose();

  if (m_ShmInfo.shmaddr)
  {
    shmdt(m_ShmInfo.shmaddr);
    m_ShmInfo.shmaddr = 0;
  }

  if (m_Image)
  {
    XFree(m_Image);
    m_Image = 0;
  }

  if (m_Display)
  {
    if (m_VideoPort)
    {
      XvUngrabPort(m_Display,m_VideoPort,1);
    }
    XDestroyWindow(m_Display,m_Window);
    if (m_Xswa.colormap)
    {
      XFreeColormap(m_Display,m_Xswa.colormap);
    }
    XCloseDisplay(m_Display);
    m_Display = 0;
  }
  printf("\nBye.\n");
}



int XVideoShow::Go()
{
  printf("starting the show...\n");
  if (OpenDisplay())
  {
    return 1;
  }
  if (OpenWindow())
  {
    return 2;
  }
  if (GetVideoPort())
  {
    return 3;
  }
  if (CreateImage())
  {
    return 4;
  }
  PlayVideo();
  
  return 0;
}



int XVideoShow::OpenDisplay()
{
  m_Display = XOpenDisplay(NULL);
  if (m_Display == NULL) 
  {
    fprintf(stderr,"cannot open display\n");
    return 1;
  }
  if (!XShmQueryExtension(m_Display)) 
  {
    fprintf(stderr,"no shmem available\n");
    return 2;
  }
  if (SelectBestDisplay())
  {
    printf("cannot use display\n");
    return 3;
  }

  return 0;
}



int XVideoShow::OpenWindow()
{
  unsigned long	mask = 0;

  // m_Xswa.colormap = XCreateColormap(m_Display,DefaultRootWindow(m_Display),m_VisInfo.visual,AllocNone);
  // mask = CWBackPixel | CWBorderPixel | CWColormap | CWEventMask;
  m_Window = XCreateWindow(m_Display,DefaultRootWindow(m_Display),0,0,m_Width,m_Height,0,m_VisInfo.depth,InputOutput,m_VisInfo.visual,mask,&m_Xswa);
  XStoreName(m_Display,m_Window,m_WindowName);
  XSetIconName(m_Display,m_Window,m_WindowName);
  XSelectInput(m_Display,m_Window,StructureNotifyMask);

  XMapWindow(m_Display,m_Window);
  XEvent event;
  do
  {
    XNextEvent(m_Display,&event);
  } while (event.type != MapNotify);

  return 0;
}



int XVideoShow::GetVideoPort()
{
  XvAdaptorInfo *adaptorInfo;
  unsigned int adaptors;
  unsigned int i;
  unsigned int j;
  int port;

  XvQueryAdaptors(m_Display,m_Window,&adaptors,&adaptorInfo);
  for( i = 0; i < adaptors; i++)
  {
    if (adaptorInfo[i].type & XvImageMask)
    {
      for (j = 0; j < adaptorInfo[i].num_ports; j++)
      {
        port = adaptorInfo[i].base_id + j;
        if (VideoPortHasYUY2(port) && (XvGrabPort(m_Display,port,0) == Success))
        {
          m_VideoPort = port;
          fprintf(stderr,"using XVIDEO adaptor %d: %s\n",port,adaptorInfo[i].name);
          XvFreeAdaptorInfo(adaptorInfo);
          return 0;
        }
      }
    }
  }

  return 1;
}



int XVideoShow::CreateImage()
{
  m_Image = XvShmCreateImage(m_Display,m_VideoPort,GUID_YUY2_PACKED,0,m_Width,m_Height,&m_ShmInfo);

  m_ShmInfo.shmid    = shmget(IPC_PRIVATE,m_Image->data_size,IPC_CREAT | 0644);
  m_ShmInfo.shmaddr  = (char *)shmat(m_ShmInfo.shmid,0,0);
  m_ShmInfo.readOnly = False;

  if (!XShmAttach(m_Display,&m_ShmInfo))
  {
    fprintf(stderr,"XShmAttach failed!\n");
    return 1;
  }
  m_Image->data = m_ShmInfo.shmaddr;
  shmctl(m_ShmInfo.shmid,IPC_RMID,0); 

  return 0;
}



int XVideoShow::PlayVideo()
{
  GC gc;
  long frames;

  MyEventsOpen(m_Display,m_Window);
  gc = XCreateGC(m_Display,m_Window,0,0);
  saved_w = w = m_Image->width; 
  saved_h = h = m_Image->height;

  for (frames = 1;; frames++)
  {
    UpdateImageData();

    if (EatEvents())
    {
      break;
    }

    //
    // show frame
    //
    XGetGeometry(m_Display,m_Window,&dw,(int *)&d,(int *)&d,&w,&h,&d,&d);
    XvShmPutImage(m_Display,m_VideoPort,m_Window,gc,m_Image,
                  0,0,m_Image->width,m_Image->height,0,0,w,h,False);
    XSync(m_Display,False);
  }
  XFreeGC(m_Display,gc);

  return 0;
}



#define TRY_DISPLAY(depth,type,text)                                  \
        if (XMatchVisualInfo(m_Display,screen,depth,type,&m_VisInfo)) \
        {                                                             \
          fprintf(stderr,"found " text " display\n");                 \
          return 0;                                                   \
        }

int XVideoShow::SelectBestDisplay()
{
  int screen = DefaultScreen(m_Display);

  TRY_DISPLAY( 24, TrueColor,   "24bit TrueColor"  )
  TRY_DISPLAY( 16, TrueColor,   "16bit TrueColor"  )
  TRY_DISPLAY( 15, TrueColor,   "15bit TrueColor"  )
  TRY_DISPLAY(  8, PseudoColor, "8bit PseudoColor" )
  TRY_DISPLAY(  8, GrayScale,   "8bit GrayScale"   )
  TRY_DISPLAY(  8, StaticGray,  "8bit StaticGray"  )
  TRY_DISPLAY(  1, StaticGray,  "1bit StaticGray"  )
 
  return -1;
}

#undef TRY_DISPLAY



int XVideoShow::VideoPortHasYUY2(XvPortID port)
{
  int ret = 0;
  XvImageFormatValues *formatValues;
  int formats;
  int i;

  formatValues = XvListImageFormats(m_Display,port,&formats);
  for (i = 0; i < formats; i++)
  {
    if ((formatValues[i].id == GUID_YUY2_PACKED) && 
        !(strcmp(formatValues[i].guid,"YUY2"))) 
    {
      ret = 1;
      break;
    }
  }
  XFree(formatValues);

  return ret;
}



int XVideoShow::EatEvents()
{
  unsigned long value;

  while (XPending(m_Display))
  {
    switch (MyEventsTrace(&value))
    {
      case KeyPress:
	   if (value == 0x20)
	   {
  	     while (1)
             {
               if (XPending(m_Display))
	       {
                 if ((MyEventsTrace(&value) == KeyPress) && (value == 0x20))
	         {
                   break;;
                 }
	       }
             }
	     break;
           }
           XGetGeometry(m_Display,m_Window,&dw,(int *)&d,(int *)&d,&w,&h,&d,&d);
           if ((saved_w != w) || (saved_h != h))
	   {
             //
             // re-aspect window 
             // 
             double area  = (double)w * h; 
             double ratio = (double)m_Image->width / m_Image->height;
             double new_w = sqrt(ratio * area);
             double new_h = area / new_w;
             saved_w = w = (unsigned int)new_w;
             saved_h = h = (unsigned int)new_h;
             XResizeWindow(m_Display,m_Window,w,h);
           }

           break;

      case ButtonPress:
	   return 1;

      default:
           break;
    }
  }

  return 0;
}



int XVideoShow::UpdateImageData()
{
  unsigned char *p = (unsigned char *)m_Image->data;
  int i;
  int j;

  //
  // Let it snow, let it snow, let it snow!
  //
  for (i = 0; i < m_Image->height; i++)
  {
    for (j = 0; j < m_Image->width; j++, p += 2) 
    {
      //
      // luminance and  "chrominance/2" - remember the YUY2 format
      //
      p[0] = 255;
      p[1] = 128;
    }
  }

  return 0;
}
