// mandrill - a filter-plugin for plantri.

// This program was written collaboratively, with contributions by 
// Stuart Anderson and Lorenz Milla. See http://www.squaring.net
// for more information on squared squares and squared rectangles.
// October 30th 2013

// To COMPILE this you must do some changes to plantri.c (version 4.5) or use the modified version plantrimod.c:
// 1. Add this line: #define PLUGIN "mandrill-plugin.c"
// 2. Change line 190 to: #define CPUTIME 0
// 3. Change line 18523 to: dosummary = 0;
// 4. Comment out the lines 18034-18087.
// Then compile like this: gcc plantrimod.c -o mandrill.exe -O4 -static -lm
// The -lm option is needed to include <math.h> in linux.

// USAGE: mandrill.exe -pc2m3v 12 v12.bin 2> man_v12.txt
// this analyzes only those graphs with f >= v (faces / vertices)
// but it counts the other ones too (f < v).
// If you don't want to count the f < v graphs you can enter a lower edge bound
// like in the plantri guide: mandrill.exe -pc2m3e22:v 12 v12.bin 2> man_v12.txt
// As the upper edge bound is not really implemented in plantri,
// you don't gain any speed if you enter an upper edge bound.
// The upper edge bound can also be done by setting the maxorder parameter (see below).

#define PLUGIN_INIT mandrill_init // loaded once in the beginning to initialize the statistics
#define FILTER(nbtot,nbop,doflip) mandrill_filter(nbtot,nbop,doflip,connec) //loaded for every graph plantri produces
#define SUMMARY mandrill_summary  // loaded once in the end to summarize the statistics

#include <time.h>
#include <math.h> // for sqrt and cbrt function

//################### Settings to be done by the user #########################

// Set the minimum sidelength of the resulting squared square.
// For perfect squared squares of order n set to ceiling(sqrt(n*(n+1)*(2*n+1)/6)) (sum of first n squares)
// If minside is higher, mandrill works faster (but doesn't find all squares with sidelength less than minside).
// In http://arxiv.org/abs/1307.2092 ("Smallest Squared Squares") Lorenz Milla proved
// that there are exactly two SISS with size up to 17x17. So if you're looking for SISS set minside to 18 -
// then only the following 16x16 SISS in order 21 isn't being found any more but mandrill is working faster.
// 21 16 16 4 5 3 4 2 1 3 1 5 2 6 5 2 3 2 5 1 4 1 4 3
int minside = 18;

// Set the maximum order up to which you want to search for squared squares.
// All graphs with more than maxorder+1 edges won't be analyzed, just counted.
// If you want to analyze all graphs that plantri produces, set maxorder to 99
int maxorder = 32;

//################## statistical variables, initialization and summary ##################

double totalzeit;            // how long the whole program was running
double analyszeit;           // how long the mandrill_filter was running
double furtheranalyszeit;    // how long the second filter was running on the graphs with good determinant

unsigned long detmax;
bigint gcount0;              // number of graphs plantri produced it total (including those not analyzed)
bigint gcount1;              // number of graphs plantri produced with at most maxorder+1 edges
bigint gcount2;              // number of graphs which passed A3
bigint gcount3;              // number of graphs which passed B5
bigint totalcount[4][MAXE];  // number of graphs with a certain number of edges produced by plantri, connectivity "exactly 2" or "3 or more"
bigint outputcount[4][MAXE]; // number of graphs with a certain number of edges saved by mandrill, connectivity "exactly 2" or "3 or more"

static void mandrill_init() { // loaded once in the beginning to initialize the statistics
  int j;
  for(j=0;j<MAXE;j++) { ZEROBIG(totalcount[2][j]); ZEROBIG(outputcount[2][j]);
                        ZEROBIG(totalcount[3][j]); ZEROBIG(outputcount[3][j]); }
  detmax=0;
  ZEROBIG(gcount0);
  ZEROBIG(gcount1);
  ZEROBIG(gcount2);
  ZEROBIG(gcount3);
  analyszeit=0.0;
  furtheranalyszeit=0.0;
  totalzeit=-clock();
}

static void mandrill_summary() { // loaded once in the end to summarize the statistics
  int me=edgebound[0]/2;
  if (me==0) {me=3*(nv-1)/2+2;}
  int i;
  totalzeit+=clock();
  fprintf(msgfile,"End of file. Maximum Determinant = %lu; settings: minside=%d, maxorder=%d\n",detmax,minside,maxorder);
  fprintf(msgfile,"total time = %.2f sec = 100 percent\n",(double)(totalzeit) / (double)(CLOCKS_PER_SEC));
  fprintf(msgfile,"time for whole analysis = %.2f sec = %.2f percent of total time\n",
        (double)(analyszeit) / (double)(CLOCKS_PER_SEC),(double)(100.0*analyszeit) / (double)(totalzeit));
  fprintf(msgfile,"time for further analysis = %.2f sec = %.2f percent of whole analysis\n",
       (double)(furtheranalyszeit) / (double)(CLOCKS_PER_SEC),(double)(100.0*furtheranalyszeit) / (double)(analyszeit));
  fprintf(msgfile,"graphs produced by plantri = "); PRINTBIG(msgfile,gcount0); fprintf(msgfile,"\n");
  fprintf(msgfile,"graphs analyzed by mandrill = "); PRINTBIG(msgfile,gcount1);
      fprintf(msgfile," = %.2f percent\n",(double)(100.0*BIGTODOUBLE(gcount1)) / (double)(BIGTODOUBLE(gcount0)));
  fprintf(msgfile,"graphs with good determinant = "); PRINTBIG(msgfile,gcount2);
     fprintf(msgfile," = %.2f percent\n",(double)(100.0*BIGTODOUBLE(gcount2)) / (double)(BIGTODOUBLE(gcount1)));
  fprintf(msgfile,"of these: graphs saved to output = "); PRINTBIG(msgfile,gcount3);
     fprintf(msgfile," = %.2f percent\n",(double)(100.0*BIGTODOUBLE(gcount3)) / (double)(BIGTODOUBLE(gcount2)));

  fprintf(msgfile,"additional graph counts by number of edges and connectivity (exactly 2-conn or at least 3-conn):\n");
  fprintf(msgfile,"edges = "); for(i=me;i<=3*maxnv-6;i++) { fprintf(msgfile,"%d ",i); } fprintf(msgfile,"\n");
  fprintf(msgfile,"totalcount2 = "); for(i=me;i<=3*maxnv-6;i++) { PRINTBIG(msgfile,totalcount[2][i]); fprintf(msgfile," "); } fprintf(msgfile,"\n");
  fprintf(msgfile,"outputcount2 = "); for(i=me;i<=3*maxnv-6;i++) { PRINTBIG(msgfile,outputcount[2][i]); fprintf(msgfile," "); } fprintf(msgfile,"\n");
  fprintf(msgfile,"totalcount3 = "); for(i=me;i<=3*maxnv-6;i++) { PRINTBIG(msgfile,totalcount[3][i]); fprintf(msgfile," "); } fprintf(msgfile,"\n");
  fprintf(msgfile,"outputcount3 = "); for(i=me;i<=3*maxnv-6;i++) { PRINTBIG(msgfile,outputcount[3][i]); fprintf(msgfile," "); } fprintf(msgfile,"\n");
}

//############### functions and variables used by mandrill_filter #############################

double K[MAXN][MAXN]; // Kirchhoff matrix K
double L[MAXN][MAXN]; // K = L * d * L^T
double d[MAXN];       // K = L * d * L^T
double A[MAXE][MAXN]; // Adjacency Matrix
double VV[MAXN][MAXN];// VV = L^(-1)
double V[MAXN][MAXN]; // V = K^(-1) * det(K)
double m[MAXE][MAXN]; // m = A * V
double F[MAXE][MAXE]; // Currents matrix F = m * A^T = A * V * A^T
double R[MAXE];       // R[i] = gcd(F[i][j])
double B[MAXE][MAXE]; // Reduced Currents Matrix B[i][j] = F[i][j] / R[i]

int gcd(int n,int m){return m==0?n:gcd(m,n%m);}

int issquare(long int n){
  long tst;
  if (n < 0) { return 0; }
  else { tst = (long)(sqrt(1.0*n)+0.5);
         if (tst*tst==n) { return 1; } else { return 0; } }
}

// Squarefree numbers: numbers that are not divisible by a square greater than 1. 
// 792 values taken from http://oeis.org/A005117/b005117.txt
// largest value: (maximum value of unsigned long / 2)^(1/3) - for larger determinants mandrill doesn't work anyway.
int sqfree[] = { 1, 2, 3, 5, 6, 7, 10, 11, 13, 14, 15, 17, 19, 21, 22, 23, 26, 29, 30, 31, 33, 34, 35, 37, 38, 39, 41, 42, 43, 46, 47, 51, 53, 55, 57, 58, 59, 61, 62, 65, 66, 67, 69, 70, 71, 73, 74, 77, 78, 79, 82, 83, 85, 86, 87, 89, 91, 93, 94, 95, 97, 101, 102, 103, 105, 106, 107, 109, 110, 111, 113, 114, 115, 118, 119, 122, 123, 127, 129, 130, 131, 133, 134, 137, 138, 139, 141, 142, 143, 145, 146, 149, 151, 154, 155, 157, 158, 159, 161, 163, 165, 166, 167, 170, 173, 174, 177, 178, 179, 181, 182, 183, 185, 186, 187, 190, 191, 193, 194, 195, 197, 199, 201, 202, 203, 205, 206, 209, 210, 211, 213, 214, 215, 217, 218, 219, 221, 222, 223, 226, 227, 229, 230, 231, 233, 235, 237, 238, 239, 241, 246, 247, 249, 251, 253, 254, 255, 257, 258, 259, 262, 263, 265, 266, 267, 269, 271, 273, 274, 277, 278, 281, 282, 283, 285, 286, 287, 290, 291, 293, 295, 298, 299, 301, 302, 303, 305, 307, 309, 310, 311, 313, 314, 317, 318, 319, 321, 322, 323, 326, 327, 329, 330, 331, 334, 335, 337, 339, 341, 345, 346, 347, 349, 353, 354, 355, 357, 358, 359, 362, 365, 366, 367, 370, 371, 373, 374, 377, 379, 381, 382, 383, 385, 386, 389, 390, 391, 393, 394, 395, 397, 398, 399, 401, 402, 403, 406, 407, 409, 410, 411, 413, 415, 417, 418, 419, 421, 422, 426, 427, 429, 430, 431, 433, 434, 435, 437, 438, 439, 442, 443, 445, 446, 447, 449, 451, 453, 454, 455, 457, 458, 461, 462, 463, 465, 466, 467, 469, 470, 471, 473, 474, 478, 479, 481, 482, 483, 485, 487, 489, 491, 493, 494, 497, 498, 499, 501, 502, 503, 505, 506, 509, 510, 511, 514, 515, 517, 518, 519, 521, 523, 526, 527, 530, 533, 534, 535, 537, 538, 541, 542, 543, 545, 546, 547, 551, 553, 554, 555, 557, 559, 561, 562, 563, 565, 566, 569, 570, 571, 573, 574, 577, 579, 581, 582, 583, 586, 587, 589, 590, 591, 593, 595, 597, 598, 599, 601, 602, 606, 607, 609, 610, 611, 613, 614, 615, 617, 618, 619, 622, 623, 626, 627, 629, 631, 633, 634, 635, 638, 641, 642, 643, 645, 646, 647, 649, 651, 653, 654, 655, 658, 659, 661, 662, 663, 665, 667, 669, 670, 671, 673, 674, 677, 678, 679, 681, 682, 683, 685, 687, 689, 690, 691, 694, 695, 697, 698, 699, 701, 703, 705, 706, 707, 709, 710, 713, 714, 715, 717, 718, 719, 721, 723, 727, 730, 731, 733, 734, 737, 739, 741, 742, 743, 745, 746, 749, 751, 753, 754, 755, 757, 758, 759, 761, 762, 763, 766, 767, 769, 770, 771, 773, 777, 778, 779, 781, 782, 785, 786, 787, 789, 790, 791, 793, 794, 795, 797, 798, 799, 802, 803, 805, 806, 807, 809, 811, 813, 814, 815, 817, 818, 821, 822, 823, 826, 827, 829, 830, 831, 834, 835, 838, 839, 842, 843, 849, 851, 853, 854, 857, 858, 859, 861, 862, 863, 865, 866, 869, 870, 871, 874, 877, 878, 879, 881, 883, 885, 886, 887, 889, 890, 893, 894, 895, 897, 898, 899, 901, 902, 903, 905, 906, 907, 910, 911, 913, 914, 915, 917, 919, 921, 922, 923, 926, 929, 930, 933, 934, 935, 937, 938, 939, 941, 942, 943, 946, 947, 949, 951, 953, 955, 957, 958, 959, 962, 965, 966, 967, 969, 970, 971, 973, 974, 977, 978, 979, 982, 983, 985, 986, 987, 989, 991, 993, 994, 995, 997, 998, 1001, 1002, 1003, 1005, 1006, 1007, 1009, 1010, 1011, 1013, 1015, 1018, 1019, 1021, 1022, 1023, 1027, 1030, 1031, 1033, 1034, 1037, 1038, 1039, 1041, 1042, 1043, 1045, 1046, 1047, 1049, 1051, 1054, 1055, 1057, 1059, 1061, 1063, 1065, 1066, 1067, 1069, 1070, 1073, 1074, 1077, 1079, 1081, 1082, 1085, 1086, 1087, 1090, 1091, 1093, 1094, 1095, 1097, 1099, 1101, 1102, 1103, 1105, 1106, 1109, 1110, 1111, 1113, 1114, 1115, 1117, 1118, 1119, 1121, 1122, 1123, 1126, 1129, 1130, 1131, 1133, 1135, 1137, 1138, 1139, 1141, 1142, 1145, 1146, 1147, 1149, 1151, 1153, 1154, 1155, 1157, 1158, 1159, 1162, 1163, 1165, 1166, 1167, 1169, 1171, 1173, 1174, 1177, 1178, 1181, 1182, 1185, 1186, 1187, 1189, 1190, 1191, 1193, 1194, 1195, 1198, 1199, 1201, 1202, 1203, 1205, 1207, 1209, 1211, 1213, 1214, 1217, 1218, 1219, 1221, 1222, 1223, 1226, 1227, 1229, 1230, 1231, 1234, 1235, 1237, 1238, 1239, 1241, 1243, 1245, 1246, 1247, 1249, 1253, 1254, 1255, 1257, 1258, 1259, 1261, 1262, 1263, 1265, 1266, 1267, 1270, 1271, 1273, 1277, 1279, 1281, 1282, 1283, 1285, 1286, 1289, 1290, 1291, 1293, 1294, 1295, 1297, 1298, 1299 };

//############### mandrill_filter ####################################

// This program finds candidates for squared squares; simple and compound, 
// perfect and imperfect squared squares are not distinguished by the software,
// by testing the determinant of the Kirchhoff matrix (discrete Laplacian
// matrix) of an input graph  to check if it can be factored as 2mk^2, 
// where m and k are integers and m is a square-free number.
// Such a factoring is a necessary but not sufficient condition for 
// a 2 or 3 connected graph to produce a squared square.
// The theory upon which Kirchhoff matrix determinant factorisation is used 
// to produce squared square candidates, is based on the 1940 paper,
// "The dissection of rectangles into squares",by Brooks,Smith,Tutte and Stone.
// In the paper, equations 2.33, 2.34 and theorem 6.12 along with the 
// definition that preceeds it give the theoretical justification.

static int mandrill_filter(int nbtot, int nbop, int doflip, int connec) {
  ADDBIG(gcount0,1);
  int neh=ne/2;
  ADDBIG(totalcount[connec][neh],1);
  if (neh>maxorder+1) { return 0; } // don't analyze graphs with more edges than maxorder+1, just count them.
  if (neh<2*nv-2) { return 0; }      // don't analyze graphs with more vertices than faces

  analyszeit-=clock();
  int i,j,k,s;
  EDGE *e,*elast;
  int a,b;
  int s1,s2;

  double det,imax,smax;
  unsigned long det_long,det2;
  int passfilter,z,gsave;
  
  //---------- PART A : Has to be done with all graphs ------------
  ADDBIG(gcount1,1);
  
  //#### A1 #### Generate full Kirchhoff Matrix K: ####

  for(i=0;i<nv;i++) {for(j=0;j<nv;j++) { K[i][j]=0.0; } }
  for (i = 0; i < nv + (missing_vertex >= 0); ++i) {
    if (i != missing_vertex) {
      e = elast = firstedge[i];
      do {
        a = e->start;
        K[a][e->end]-=1.0;
        K[a][a]++; // plantri has every edge twice (both ways), so we analyze only half of the edge
        e = e->next;
      } while (e!=elast);
    }
  }

  //#### A2 #### Get determinant of Kirchhoff Matrix by LDL-decomposition: ####
    
  det=1.0;
  for (j=0;j<nv-1;j++) {
    L[j][j]=1.0;
    d[j]=K[j][j];
    for(k=0;k<j;k++) { d[j] = d[j] - L[j][k]*d[k]*L[j][k]; }
    for(i=j+1;i<nv-1;i++) {
      L[j][i]=0;
      L[i][j]=K[i][j];
      for(k=0;k<j;k++) { L[i][j] = L[i][j] - L[i][k]*d[k]*L[j][k]; }
      L[i][j] = L[i][j]/d[j];
    }
    det=det*d[j];
  }

  //#### A3 #### Check if det = 2 * m * k^2 for k>=minside ####

  passfilter=0;
  det_long = (long unsigned)(det+0.5);
  if (det_long>detmax) { detmax=det_long; }
  if (det_long % 2 == 0) {
    det2 = det_long/2;
    imax = (long unsigned)(cbrt(0.5*det)+0.5); // imax = (det/2)^(1/3)
    for(i=minside;i<imax;i++) {
      if((det2 % (i*i))==0) { passfilter=1; break; }
    }
    if(passfilter==0) {
      smax=0.5*det/(1.0*(imax-1)*(imax-1)); // this line was changed in Version 2 (Bug fixed).
      for(s=0;sqfree[s]<=smax;s++) {
        if(det2 % sqfree[s] == 0) {
          if (issquare(det2/sqfree[s])==1) { passfilter=1; break; }
        }
      }
    }
  }
  
  gsave = 0;

  //---------- PART B : Just the graphs who passed A3 ------------

  if(passfilter==1) {
    furtheranalyszeit-=clock();
    ADDBIG(gcount2,1);

    //#### B1 #### Generate the Adjacency Matrix A: ####

    for(i=0;i<neh;i++) { for(j=0;j<nv;j++) { A[i][j]=0.0; } }
    k=0;
    for(i=0;i<nv;i++) {
      for(j=i+1;j<nv;j++) {
        if(K[i][j]+1.0<0.001) { A[k][i]=1.0; A[k][j]=-1.0; k++; }
      }
    }

    //#### B2 #### Invert the Kirchhoff matrix: V = K^(-1)*det(K) ####

    for(i=0;i<nv-1;i++) {
      for(j=0;j<nv-1;j++) {
       if(i==j) { VV[i][j]=1.0; } else { VV[i][j]=0.0; }
      }
    }
    for(i=0;i<nv-1;i++) {
      for(j=i+1;j<nv-1;j++) {
        for(k=0;k<=j;k++) {
          VV[j][k]-=L[j][i]*VV[i][k];
        }
      }
    } // now VV=L^(-1)

    for(i=0;i<nv-1;i++) {
      for(j=0;j<=i;j++) {
        V[i][j]=0.0;
        for(k=i;k<nv-1;k++) {
          V[i][j]+=VV[k][i]*det/d[k]*VV[k][j];
        }
        V[j][i]=V[i][j];
      }
    } // now V = K^(-1)*det(K)

    //#### B3 #### calculate the currents matrix F = A * V * A^T ####

    for(i=0;i<neh;i++) {
      for(j=0;j<nv-1;j++) {
        m[i][j]=0.0;
        for(k=0;k<nv-1;k++) {
          m[i][j]+=A[i][k]*V[k][j];
        }
      }
    } // now m = A * V

    for(i=0;i<neh;i++) {
      for(j=0;j<neh;j++) {
        F[i][j]=0.0;
        for(k=0;k<nv-1;k++) {
          F[i][j]+=m[i][k]*A[j][k];
        }
      }
    } // now F = m * A^T = A * V * A^T

    //#### B4 #### calculate the currents reduction factors vector R ####
    //############ and the reduced currents matrix B ####

    for(i=0; i<neh; i++) {
      if ((long)(F[i][i]+0.5)!= 0){
        R[i] = abs(gcd((long)(F[i][0]+0.5),(long)(F[i][1]+0.5)));
      }
      else R[i]=1;
      for (j=1; j<neh; j++){
        if ((long)(F[i][j]+0.5)!= 0){
          R[i] = abs(gcd((long)(R[i]+0.5), (long)(F[i][j]+0.5)));
        }
      }
    }

    for(i=0; i<neh; i++) {
      for(j=0; j<neh; j++) {
        B[i][j] = (long)(F[i][j]+0.5)/R[i];
        if (abs(B[i][j])<0.0001){ B[i][j] = 0.0; }
      }
    } // now B = F/R

    //#### B5 #### Test if it is a squared square with no zero currents ####

    for(i=0; i<neh; i++) {
      s1 = (long)(B[i][i]+0.5); // side1 of the tiling is B(i,i);
      s2 = det_long/(long)(R[i]+0.5)-s1; //side2 = determinant/reduction - side1
      z=1; // z = 0; no zero edges, z = 1; has zero edges
      if(s1==s2) {
        z = 0;
        for(j=0; j<neh; j++){  
          if (abs(B[i][j]) < 0.0001) { z = 1; break; }
        }
      }
      if (z == 0) { gsave=1; ADDBIG(gcount3,1); break; } // if no zero edge: save graph (gsave=1).
    }
    furtheranalyszeit+=clock();
  }
  ADDBIG(outputcount[connec][neh],gsave);
  analyszeit+=clock();
  return gsave;
}

