#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <math.h>
#include "chol.h"

#define BLOCK_SIZE 10000
#define LIMIT  33

// protos
void create_joint_graph( char* graph1,
                         char* graph2,
                         bool  swap,

                         int   graph[SIZE][SIZE],
                         int&  num_nodes,
                         int   branch_count[SIZE] );
void create_matrix( int graph[SIZE][SIZE], int num_nodes, int num_branches[SIZE], double m[SIZE][SIZE] );
void print_graph( int graph[SIZE][SIZE], int num_nodes, int current_branches_per_node[SIZE] );
double solve_mat3( double M00, double M11, double M22, double M10, double M20, double M21 );
double gcd( double x, double y );
void print_square( int graph[SIZE][SIZE], int num_nodes, int num_branches[SIZE], double vnode[SIZE], double det );
void solve_mat3_ext( double M00, double M11, double M22, double M10, double M20, double M21,
                        double& i0, double& i1, double& i2 );




int main( int argc, char* argv[] )
{
#ifdef DEBUG
    double count_0 = 0.;    // 2*input_lines*buffer_lines
    double count_1 = 0.;    // count that passes rmin/rmax testing
    double count_2 = 0.;    // count that passes R matrix test
    double count_3 = 0.;    // count that passes I unequal test
    double count_4 = 0.;    // actual perfect squares
    double overlap_1 = 0.;
    double overlap_4 = 0.;
    double range_1 = 0.;
    double range_4 = 0.;
    double count_0_threshold = 0.;
#endif
    if ( argc != 2 ) {
        fprintf( stderr, "need buffer size\n" );
        return 1;
    }
    int buffer_size = 0;
    sscanf( argv[1], "%d", &buffer_size );
    fprintf( stderr, "buffer_size=%d\n", buffer_size );

    struct Network {
        int  order;
        char resistance_matrix[50];
        char graph[ SIZE ];
        double r00;
        double r11;
        double r22;
        double r10;
        double r20;
        double r21;
        double rmin;
        double rmax;
        char  sex;
    };
    
    Network* network_buffer = new Network[ buffer_size ];

    // variables to read from input
    int order;
    int terminal_count;
    char sex[10];
    char resistance_matrix[SIZE];
    int rmin;
    int rmax;
   // int num_nodes;
    char graph[SIZE];
    int max_rmax = 0;

    for ( int i=0; ; ++i) {
        int n = scanf("%d %d %s %s %d %d %d %s\n",
                    &order,
                    &terminal_count,
                    sex,
                    resistance_matrix,
                    &rmin,
                    &rmax,
                    &order,
                    graph );

        if ( n == EOF ) {
            return 0;
        }
        assert( n == 8 );
        assert( *sex=='M' || *sex=='F' );
        assert( strlen(sex) < sizeof(sex) );
        // currently we only handle 3 terminal networks.
        assert( terminal_count == 3 );
        assert( strlen( resistance_matrix ) < sizeof( resistance_matrix ) );
        assert( strlen( graph ) < sizeof( graph ) );

        int n1,n2,n3,n4,n5,n6,d;
        int x = sscanf( resistance_matrix, "%d,%d,%d,%d,%d,%d,%d", &n1, &n2, &n3, &n4, &n5, &n6, &d );
        assert( x == 7 );
        double r00;
        double r10;
        double r11;
        double r20;
        double r21;
        double r22;

        double dr = 1. / double(d);
        r00 = double(n1)*dr;
        r10 = double(n2)*dr;
        r11 = double(n3)*dr;
        r20 = double(n4)*dr;
        r21 = double(n5)*dr;
        r22 = double(n6)*dr;

        if ( i < buffer_size ) {
            // put input data into buffer
            Network* n = &network_buffer[i];
            n->order = order;
            n->r00 = r00;
            n->r10 = r10;
            n->r11 = r11;
            n->r20 = r20;
            n->r21 = r21;
            n->r22 = r22;
            n->rmin = rmin;
            n->rmax = rmax;
            strcpy( n->graph, graph );
            n->sex = *sex;
            if ( rmax > max_rmax ) {
                max_rmax = rmax;
            }
        } else {
            if ( i == buffer_size ) {

                Network* buf = network_buffer;

                // run quicksort algorithm on the buffer
                const int SORT_LIST_SIZE = 100;
                struct SortListEntry {
                    int min;
                    int max;
                } sort_list[ SORT_LIST_SIZE ];
                int sort_list_count = 1;
                sort_list[0].min = 0;
                sort_list[0].max = buffer_size - 1;

                while ( sort_list_count > 0 ) {
                    int min = sort_list[ sort_list_count-1 ].min;
                    int max = sort_list[ sort_list_count-1 ].max;
                    assert( max > min );
                    // sort all the elements in buf between min and max
                    Network* p = &buf[min];
                    Network* pivot = &buf[ ( min + max )/2 ];

                    // swap pivot and max
                    Network temp = buf[max];
                    buf[max] = *pivot;
                    *pivot = temp;
                    pivot = &buf[max];

                    for ( Network* q = p; q < &buf[max]; ++q ) {
                        if  (  q->rmin < pivot->rmin ) {
                            // swap q with p and increment p.
                            temp = *q;
                            *q = *p;
                            *p = temp;
                            ++p;
                        }
                    }
                    
                    // swap p with pivot
                    temp = *pivot;
                    *pivot = *p;
                    *p = temp;

                    // find the two sublists that need to be sorted
                    int min_lower = min;
                    int max_lower = p - 1 - &buf[0];
                    int min_upper = p + 1 - &buf[0];
                    int max_upper = max;

                    // remove list just sorted from list
                    --sort_list_count;

                    // add the larger of the two counts to the list first
                    if ( max_lower - min_lower >= max_upper - min_upper && max_lower - min_lower > 0 ) {
                        sort_list[ sort_list_count ].min = min_lower;
                        sort_list[ sort_list_count ].max = max_lower;
                        ++sort_list_count;

                        if ( max_upper - min_upper > 0 ) {
                            sort_list[ sort_list_count ].min = min_upper;
                            sort_list[ sort_list_count ].max = max_upper;
                            ++sort_list_count;
                        }
                    } else {
                        if ( max_upper - min_upper > 0 ) {
                            sort_list[ sort_list_count ].min = min_upper;
                            sort_list[ sort_list_count ].max = max_upper;
                            ++sort_list_count;
                        }
                        if ( max_lower - min_lower > 0 ) {
                            sort_list[ sort_list_count ].min = min_lower;
                            sort_list[ sort_list_count ].max = max_lower;
                            ++sort_list_count;
                        }
                        assert( sort_list_count <= SORT_LIST_SIZE );
                    }
                }

                // do a double check that this lists are sorted properly.
                for ( int j = 0; j < buffer_size-1; ++j ) {
                    assert( network_buffer[j+1].rmin >= network_buffer[j].rmin );
                }
            }
                                

            // check if value of rmax excludes this network
#ifdef DEBUG
            count_0 += 2*buffer_size;
            if ( count_0 >= count_0_threshold ) {
                count_0_threshold += 1e9;
                fprintf( stderr, "count_0 = %g\n", count_0 );
                fprintf( stderr, "count_1 = %g %g\n", count_1, count_1/count_0 );
                fprintf( stderr, "count_2 = %g %g\n", count_2, count_2/count_1 );
                fprintf( stderr, "count_3 = %g %g\n", count_3, count_3/count_2 );
                fprintf( stderr, "count_4 = %g %g %g\n", count_4, count_4/count_3, count_4/count_1 );
                fprintf( stderr, "av range = %g  %g\n", range_1/count_1, range_4/count_4 );
                fprintf( stderr, "av overlap = %g  %g\n\n", overlap_1/count_1, overlap_4/count_4 );
                fflush( stderr );
            }
#endif
            if ( rmax + max_rmax < ( 1<<24 ) ) {
                continue;
            }

            // do a binary search on rmin so we only check connections with possible networks
            int a = 0;
            int b = buffer_size - 1;
            if ( rmin + network_buffer[0].rmin > ( 1<<24 ) ) {
                // none of the buffered networks will work
                continue;
            }
            for ( int c = ( a + b ) / 2; b > a+1; c = ( a + b ) / 2 ) {
                if ( rmin + network_buffer[c].rmin > ( 1<< 24 ) ) {
                    b = c;
                } else {
                    a = c;
                }
            }

            for ( int j = 0; j < b; ++j ) {

                // try the network we just read with all networks in buffer which pass the rmin test.
                Network* n = &network_buffer[j];

                if ( rmax + n->rmax < 1<<24 ) {
                    continue;
                }

#ifdef DEBUG
                count_1 += 2.;
                double range = ( rmax - rmin + n->rmax - n->rmin ) / double( 1<<24 );
                range_1 += 2. * range;
                double overlap = ( range - fabs( 2. - ( rmin + rmax + n->rmin + n->rmax )/ double(1<<24) ) ) / range;
                overlap_1 += 2 * overlap;
#endif

                int combined_graph[SIZE][SIZE];
                int num_nodes;
                int num_branches[SIZE];
                for ( int swap = 0; swap < 2; ++swap ) {
                    // connect in two ways. if swap, reflect one network with respect to the other.
                    double rr;
                    if ( swap ) {
                        rr = 1./ solve_mat3(     
                                        r00 + n->r22,
                                        r11 + n->r11,
                                        r22 + n->r00,
                                        r10 + n->r21,
                                        r20 + n->r20,
                                        r21 + n->r10
                                );
                    } else {
                        rr = 1./ solve_mat3(     
                                        r00 + n->r00,
                                        r11 + n->r11,
                                        r22 + n->r22,
                                        r10 + n->r10,
                                        r20 + n->r20,
                                        r21 + n->r21
                                );
                    }
                    if ( fabs( rr - 1.0 ) < 1e-9 ) {

#ifdef DEBUG
                        count_2 += 1.;
#endif
                        double i0;
                        double i1;
                        double i2;
                        // solve the equations a second time to be sure.
                        if ( swap ) {
                            solve_mat3_ext(     
                                        r00 + n->r22,
                                        r11 + n->r11,
                                        r22 + n->r00,
                                        r10 + n->r21,
                                        r20 + n->r20,
                                        r21 + n->r10,

                                        i0,
                                        i1,
                                        i2
                                );
                        } else {
                            solve_mat3_ext(     
                                        r00 + n->r00,
                                        r11 + n->r11,
                                        r22 + n->r22,
                                        r10 + n->r10,
                                        r20 + n->r20,
                                        r21 + n->r21,

                                        i0,
                                        i1,
                                        i2
                                );
                        }
                        if ( fabs( i0-i1 ) < 1e-10 || fabs( i1-i2 ) < 1e-10  || fabs( i0-i2 ) < 1e-10 ) {
                            // it is still possible this is a perfect square, but very unlikely.
                            // In fact it is not worth the time to investigate further.
                            // This test weeds out enough cases to measurably speed up the search.
                            continue;
                        }

#ifdef DEBUG
                        count_3 += 1.;
#endif
                        create_joint_graph( graph, n->graph, swap, 
                                            combined_graph, num_nodes, num_branches );
                        // print_graph( graph, num_nodes, num_branches );
                        double m[SIZE][SIZE];
                        create_matrix( combined_graph, num_nodes, num_branches, m );
                        // print_matrix( m, num_nodes - 1 );
                        double det = inverse( m, num_nodes - 1 );
                        // print_matrix( m, num_nodes - 1 );

                        assert( fabs( rr - m[0][0] < 1e-6 ) );
                        assert( rr * double(1<<24) < double(rmax) + double(n->rmax)  );
                        assert( rr * double(1<<24) > double(rmin) + double( n->rmin ) );

                        double vnode[SIZE];
                        vnode[0] = 0;

                        // printf( "VNODES: 0  ");
                        for ( int i = 0; i < num_nodes; ++i ) {
                            vnode[i+1] = floor( det * m[i][0] + 0.5 );
                            // printf( "%g  ", vnode[i+1] );
                        }
                        // printf( "\n");
                        double sq[SIZE];
                        int num_squares = 0;
                        for ( int i = 0; i < num_nodes; ++i ) {
                            for ( int j = 0; j<num_branches[i]; ++j ) {
                                int cnode = combined_graph[i][j];
                                if ( cnode > i ) {
                                    sq[ num_squares++ ] = fabs( vnode[cnode] - vnode[i] );
                                }
                            }
                        }
                        bool perfect = true;
                        bool zero = false;
                        bool not_square = (  floor( det*rr + 0.5 ) != floor( det + 0.5 ) );
                        double imperfect;
                        for ( int i = 0; i < num_squares; ++ i ) {
                            for ( int j = 0; j < i; ++j ) {
                                if ( sq[i] == 0. ) {
                                    zero = true;
                                    perfect = false;
                                }
                                if ( sq[i] == sq[j] ) {
                                    perfect = false;
                                    imperfect = sq[i];
                                }
                            }
                        }
                        if ( not_square ) {
                            // print to stderr
                            double gg = floor( det + 0.5 );
                            for ( int i = 0; i < num_squares; ++i ) {
                                gg = gcd( gg, sq[i] );
                            }
                            fprintf( stderr, "Not square %12f %12f \n", floor( det/gg + 0.5 ),  floor( vnode[1]/gg + 0.5 ) );
                            continue;
                        }
                        if ( zero ) {
                            //fprintf( stderr, "square with zeroes\n" );
                            continue;
                        }
                        if ( !perfect ) {
                            // fprintf( stderr, "square not perfect\n" );
                            continue;
                        }
                        // reduce square size
                        //fprintf( stderr, "square perfect rmin=%d rmax=%d\n", rmin+n->rmin, rmax+n->rmax  );
                        print_square( combined_graph, num_nodes, num_branches, vnode, det );
#ifdef DEBUG
                        count_4 += 1.;
                        range_4 += range;
                        overlap_4 += overlap;
#endif
                    }
                }
            }
        }
    }
    printf( "END\n" );
}


void create_joint_graph( char* graph1,
                         char* graph2,
                         bool  swap,

                         int   graph[SIZE][SIZE],
                         int&  num_nodes,
                         int   branch_count[SIZE] )
{
    // count nodes in each graph
    int max_node1 = 0;
    int max_node2 = 0;

    for ( char* p = graph1; *p; ++p ) {
        if ( *p == ',' ) {
            ++max_node1;
        }
    }
    for ( char* p = graph2; *p; ++p ) {
        if ( *p == ',' ) {
            ++max_node2;
        }
    }

    // set up node mappings from graph 1 and 2 to common graph
    int node_g1_to_com[SIZE];
    int node_g2_to_com[SIZE];

    // map grounds to 0 and 1
    node_g1_to_com[0] = 0;
    node_g2_to_com[0] = 1;

    // map other g1 nodes consecutively
    for ( int i = 1; i<= max_node1; ++i ) {
        node_g1_to_com[i] = i+1;
    }

    // nodes 1-3 of g2 are mapped conditionally
    int next_common_node = max_node1 + 2;
    node_g2_to_com[1] = swap ? 4 : 2;
    node_g2_to_com[2] = 3;
    node_g2_to_com[3] = swap ? 2 : 4;

    // map remaining g2 nodes sequentially
    for( int i = 4; i<= max_node2; ++i ) {
        node_g2_to_com[i] = next_common_node++;
    }
    num_nodes = next_common_node;

    // parse graph 1 and add nodes
    for ( int i = 0; i < num_nodes; ++i ) {
        branch_count[i] = 0;
    }
    for ( int graph_num = 1; graph_num <= 2; ++ graph_num ) {
        int cur_node = 0;
        char * cur_graph = ( graph_num == 1 ) ? graph1 : graph2;
        int*  node_g_to_com = ( graph_num == 1 ) ? node_g1_to_com : node_g2_to_com;
        bool reverse_add = graph_num == 2 && !swap;
        char* p = cur_graph;
        do {
            int com_node = node_g_to_com[ cur_node ];
            if ( reverse_add ) {
                while ( *p != ',' && *p != 0 ) {
                    ++p;
                }
                char *q = p;
                --p;
                while (  p >= cur_graph && *p != ',' ) {
                    int n = *p-- - 'a';
                    graph[ com_node ][ branch_count[ com_node ]++ ] =  node_g_to_com[ n ];
                }
                p = q;
            } else {
                while ( *p != ',' && *p != 0) {
                    int n = *p++ - 'a';
                    graph[ com_node ][ branch_count[ com_node ]++ ] =  node_g_to_com[ n ];
                }
            }
            ++cur_node;
        } while ( *p++ );
    }
}


void create_matrix( int graph[SIZE][SIZE], int num_nodes, int num_branches[SIZE], double m[SIZE][SIZE] )
{
    int N = num_nodes-1;
    for ( int i = 0; i<N; ++i ) {
        for( int j = 0; j<=i; ++j ) {
            m[i][j] = 0.;
        }
    }

    for ( int i = 0; i<N; ++i ) {
        for ( int j = 0; j<num_branches[i]; ++j ) {
            int k = graph[i][j];
            if ( k > i ) {
                if ( k > 0  && i > 0 ) {
                    m[k-1][i-1] += -1.;
                }
                if ( k > 0 ) {
                    m[k-1][k-1] += 1.;
                }
                if ( i > 0 ) {
                    m[i-1][i-1] += 1.;
                }
            }
        }
    }
}


void print_graph( int graph[SIZE][SIZE], int num_nodes, int current_branches_per_node[SIZE] )
{
    printf("GGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGG");
    for (int i=0; i<num_nodes; ++i ) {
        printf("\n%3d:", i );
        for (int j = 0; j<current_branches_per_node[i]; ++j ) {
            printf(" %3d", graph[i][j] );
        }
    }
    printf("\n\n");
}





double solve_mat3( double M00, double M11, double M22, double M10, double M20, double M21 )
{
    // the return value is the sum of all elements of the inverse.
    // alternatively solve M*X = [1,1,1], then sum elements of X.

    // M = L * D * Lt
   // double D00 = M00;
    double D00r = 1. / M00;
    double L10 = M10*D00r;
    double L20 = M20*D00r;

    // M11 = L10*D00*L10 + D11
    double D11 = M11 - M10*L10;

    // M21 = L20*D00*L10 + L21*D11
    double L21xD11 =  M21 - L20*M10;
    double L21 = L21xD11 / D11;

    // M22 = L20*D00*L20 + L21*D11*L21 + D22
    double D22 = M22 - L20*M20 - L21*L21xD11;
   // double D22r = 1. / D22;

    // #######################
    // L*D*Lt*X = [1,1,1]
    // Y = Lt*X
    // Z = D*Lt*X

    // L*Z = [1,1,1];
    // double Z0 = 1.;
    // double Z1 = 1. - L10*Z0;
    // double Z2 = 1. - L20*Z0 - L21*Z1
    double Z1 = 1. - L10;
    double Z2 = 1. - L20 - L21*Z1;

    // D*Y = Z
    double Y0 = D00r;
    double Y1 = Z1 / D11;
    double Y2 = Z2 / D22;

    // Lt*X = Y
    double X2 = Y2;
    double X1 = Y1 - X2*L21;
    double X0 = Y0 - X1*L10 - X2*L20;

    return X0 + X1 + X2;
}

void solve_mat3_ext( double M00, double M11, double M22, double M10, double M20, double M21,
                        double& i0, double& i1, double& i2 )
{
    // the return value is the sum of all elements of the inverse.
    // alternatively solve M*X = [1,1,1], then sum elements of X.

    // M = L * D * Lt
   // double D00 = M00;
    double D00r = 1. / M00;
    double L10 = M10*D00r;
    double L20 = M20*D00r;

    // M11 = L10*D00*L10 + D11
    double D11 = M11 - M10*L10;

    // M21 = L20*D00*L10 + L21*D11
    double L21xD11 =  M21 - L20*M10;
    double L21 = L21xD11 / D11;

    // M22 = L20*D00*L20 + L21*D11*L21 + D22
    double D22 = M22 - L20*M20 - L21*L21xD11;
   // double D22r = 1. / D22;

    // #######################
    // L*D*Lt*X = [1,1,1]
    // Y = Lt*X
    // Z = D*Lt*X

    // L*Z = [1,1,1];
    // double Z0 = 1.;
    // double Z1 = 1. - L10*Z0;
    // double Z2 = 1. - L20*Z0 - L21*Z1
    double Z1 = 1. - L10;
    double Z2 = 1. - L20 - L21*Z1;

    // D*Y = Z
    double Y0 = D00r;
    double Y1 = Z1 / D11;
    double Y2 = Z2 / D22;

    // Lt*X = Y
    double X2 = Y2;
    double X1 = Y1 - X2*L21;
    double X0 = Y0 - X1*L10 - X2*L20;

    i0 = X0;
    i1 = X1;
    i2 = X2;
}


// round to integer and assert that value is almost an integer.
double round( double x ) 
{
    double r = floor ( x + 0.5 );
    assert( fabs( r - x ) < 0.01 );
    return r;
}

void print_square( int graph[SIZE][SIZE], int num_nodes, int num_branches[SIZE], double vnode[SIZE], double det )
{
    // find greatest common divisor
    double gg = round( det );
    for ( int i = 0; i < num_nodes; ++i ) {
        gg = gcd( gg, round( vnode[i] ) );
    }
    double total_size = round( det / gg );

    // normalize vnode
    for ( int i = 0; i < num_nodes; ++i ) {
        vnode[i] = round( vnode[i] / gg );
    }

    // compute size, horizontal and vertical position of each square
    double sqsize[ SIZE ];
    double sqh[ SIZE ];
    double sqv[ SIZE ];
    int node1[ SIZE ];
    int node2[ SIZE ];
    int num_squares = 0;

    bool node_expanded[ SIZE ];
    memset( node_expanded, 0 , sizeof( node_expanded ) );

    for ( int i = 0; i < num_nodes; ++i ) {
        // loop through nodes in acending voltage order
        double v = 2.* total_size;
        int node = -1;
        for ( int j = 0; j < num_nodes; ++j ) {
            if ( !node_expanded[j] && vnode[j] < v ) {
                node = j;
                v = vnode[j];
            }
        }
        assert( node >= 0 );
        node_expanded[ node ] = true;

        // find left most new an old node.
        int left_new = -1;
        int left_old = -1;
        int nb = num_branches[ node ];
        v = vnode[ node ];
        for ( int j = 0; j < nb; ++j ) {
            double thisv = vnode[ graph[node][j] ] - v;
            double nextv = vnode[ graph[node][(j+1)%nb] ] - v;
            if ( thisv < 0. && nextv > 0. ) {
                left_new = (j+1)%nb;
                left_old = j%nb;
                break;
            }
        }

        if ( left_new == -1 ) {
            left_new = 0;
        }

        double horizontal;
        if ( left_old == - 1 ) {
            horizontal = 0.;
        } else {
            int left_old_node = graph[ node ][ left_old ];
            assert( node_expanded[ left_old_node ] );
            // find the square corresponding to left_ld
            horizontal = -1.;
            for ( int j = 0; j < num_squares; ++j ) {
                if ( node1[j] == left_old_node && node2[j] == node ) {
                    // found it.
                    horizontal = sqh[ j ];
                    break;
                }
            }
            assert( horizontal >= 0. );
        }

        double vertical = vnode[node];

        for ( int j = left_new; j < left_new + nb; ++j ) {
            int k  = j % nb;
            int connecting_node = graph[node][k];
            if (  vnode[ connecting_node ] < v ) {
                break;
            }
            int current_square = num_squares++;
            sqsize[ current_square ] = vnode[ connecting_node ] - v;
            node1[ current_square ] = node;
            node2[ current_square ] = connecting_node;
            sqv[ current_square ] = vertical;
            sqh[ current_square ] = horizontal;
            horizontal += sqsize[ current_square ];
        }
    }

    // switch to preferred orientation
    for ( int xx=0; xx < 2; ++xx ) {
        // run this twice - second time check only.
        int corner_00 = -1;
        int corner_v = -1;
        int corner_h = -1;
        int corner_vh = -1;
        for ( int i = 0; i < num_squares; ++i ) {
            if ( sqv[i] == 0 && sqh[i] == 0 ) {
                corner_00 = i;
            }
            else if ( sqv[i] + sqsize[i] == total_size && sqh[i] == 0 ) {
                corner_v = i;
            } else if ( sqv[i] == 0 && sqh[i] + sqsize[i] == total_size ) {
                corner_h = i;
            } else if ( sqv[i] + sqsize[i] == total_size && sqh[i] + sqsize[i] == total_size ) {
                corner_vh = i;
            }
        }
        assert( corner_00 >= 0 );
        assert( corner_v >= 0 );
        assert( corner_h >= 0 );
        assert( corner_vh >= 0 );
        int size00 = sqsize[ corner_00 ];
        int sizeh = sqsize[ corner_h ];
        int sizev = sqsize[ corner_v ];
        int sizevh = sqsize[ corner_vh ];
        if ( xx == 1 ) {
            // run check and break.
            assert( size00 > sizeh );
            assert( size00 > sizev );
            assert( size00 > sizevh );
            assert( sizeh > sizev );
            break;
        }
        bool swaph = false;
        bool swapv = false;
        bool swap_axis = false;
        if ( sizeh > size00 && sizeh > sizev && sizeh > sizevh ) {
            swaph = true;
            if ( sizevh > size00 ) {
                swap_axis = true;
            }
        } else if ( sizev > size00 && sizev > sizeh && sizev > sizevh ) {
            swapv = true;
            if ( size00 > sizevh ) {
                swap_axis = true;
            }
        } else if ( sizevh > size00 && sizevh > sizeh && sizevh > sizev ) {
            swapv = true;
            swaph = true;
            if ( sizeh > sizev ) {
                swap_axis = true;
            }
        } else {
            if ( sizev > sizeh ) {
                swap_axis = true;
            }
        }
        // do the swaps
        for ( int i = 0; i < num_squares; ++i ) {
            if ( swaph ) {
                sqh[i] = total_size - sqh[i] - sqsize[i];
            }
            if ( swapv ) {
                sqv[i] = total_size - sqv[i] - sqsize[i];
            }
            if ( swap_axis ) {
                double temp = sqh[i];
                sqh[i] = sqv[i];
                sqv[i] = temp;
            }
        }
    }



    // at this point we should know the horizontal and vertical position of each square;
    // run some checks to insure no errors.
    assert( total_size == floor( total_size ) );
    double sum_square = 0.;
    for ( int i = 0; i < num_squares; ++i ) {
        double s = sqsize[i];
        double h = sqh[i];
        double v = sqv[i];
        sum_square += s*s;
        assert( s == floor(s) );
        assert( h == floor(h) );
        assert( v == floor(v) );
        assert( h >= 0. && v >= 0 );
        assert( h + s <= total_size );
        assert( v + s <= total_size );
        // check no overlap with other squares
        for ( int j = 0; j < num_squares; ++j ) {
            if ( i == j ) {
                continue;
            }
            double ss = sqsize[j];
            double hh = sqh[j];
            double vv = sqv[j];
            assert(        v >= vv + ss
                        || v + s <= vv
                        || h >= hh + ss
                        || h + s <= hh );
        }
    }
    assert( sum_square == total_size * total_size );

    // everything looks good
    printf("%d %g %g (", num_squares, total_size, total_size );
    bool printed[SIZE];
    memset( printed, 0, sizeof(printed) );

    // print squares 
    double vprev;
    double hprev;
    double sprev;
    for ( int i = 0; i < num_squares; ++i ) {
        // find next square to print
        double v = total_size;
        double h = total_size;
        int n = -1;
        for ( int j = 0; j < num_squares; ++j ) {
            if ( !printed[j] && ( sqv[j] < v || ( sqv[j] == v && sqh[j] < h ) ) ) {
                n = j;
                v = sqv[j];
                h = sqh[j];
            }
        }
        assert( n >= 0 );
        // print separator
        if ( i > 0 ) {
            if ( v == vprev && h == hprev + sprev ) {
                printf(",");
            } else {
                printf(")(");
            }
        }
        printf("%g", sqsize[n] );
        printed[n] = true;
        vprev = v;
        hprev = h;
        sprev = sqsize[n];
    }
    printf( ")\n" );
    fflush( stdout );
}













