#include <stdio.h>
#include <assert.h>
#include <string.h>
#include <math.h>


// Protos
#define SIZE 100
void add_edge_recursive( int current_node, int current_edge );
void add_edge( int node1, int edge1, int node2, int edge2 );
void remove_edge( int node1, int edge1, int node2, int edge2 );
void find_next_edge_on_face( int current_node, int current_edge, int *next_node, int *next_edge );
bool screen_automorphism( );
bool search_automorphisms( );
void print_graph( );
bool check_2_connected( int num_nodes,
                        int graph[SIZE][SIZE],
                        int edges_per_node[SIZE]  );


// find all simple connected planar graphs with at least 3 edges connecting to each node.
// a callback is generated for each graph found.  One from each isomorphism class is returned.
// Reflections may be optionally suppressed.

// use some static variables to minimize argument passing
static void (*callback_s)( int   num_nodes,
                           int   graph[SIZE][SIZE],
                           int   edges_per_node[SIZE]  );
static int graph[SIZE][SIZE];
static int edges_per_node[SIZE];
static int num_nodes_s;
static int num_edges_s;
static int sup_refl_s;

void find_graphs(   int     num_nodes,
                    int     num_edges,
                    bool    suppress_reflections,
                    void    (*callback)( int   num_nodes,
                                         int   graph[SIZE][SIZE],
                                         int   edges_per_node[SIZE]  ) )
{
    callback_s = callback;
    num_nodes_s = num_nodes;
    num_edges_s = num_edges;
    sup_refl_s = suppress_reflections;
    // initialize graph to contain a single edge from node-0 to node-1;
    graph[0][0] = 1;
    graph[1][0] = 0;
    edges_per_node[0] = 1;
    edges_per_node[1] = 1;
    for ( int i = 2; i<= num_nodes; ++i ) {
        edges_per_node[i] = 0;
    }
    int first_node_to_search = 0;
    int first_edge_to_search = 1;
    add_edge_recursive( first_node_to_search, first_edge_to_search );
}


void add_edge_recursive( int current_node, int current_edge )
{
    if ( edges_per_node[ current_node ] == 0 ) {
        // this would create a disconnection.
        return;
    }

    // require node 0 have maximum number of edges
    if ( edges_per_node[ current_node ] > edges_per_node[ 0 ] ) {
        return;
    }

    // require node 1 have maximum edges of those connecting to node 0
    if ( current_node > 1 && current_node <= edges_per_node[ 0 ]
            && edges_per_node[ current_node ] > edges_per_node[ 1 ] ) {
        return;
    }

    // make sure there are enough edges and nodes left to complete the graph
    {
        int edge_ends_used = 0;
        int edge_ends_needed = 0;
        for ( int i = 0; i< num_nodes_s; ++i ) {
            edge_ends_used += edges_per_node[ i ];
            if (  edges_per_node[ i ] < 3 ) {
                edge_ends_needed += 3 - edges_per_node[ i ];
            }
        }
        if ( edge_ends_needed > 2*num_edges_s - edge_ends_used ) {
            // not enough edges remaining to fill out nodes.
            return;
        }
        int remaining_nodes = num_nodes_s - current_node;
        int max_new_edge_ends = remaining_nodes * ( remaining_nodes - 1 );
        if ( max_new_edge_ends < 2*num_edges_s - edge_ends_used ) {
            // not enough nodes left to consume needed edges.
            return;
        }
    }

    // find the first unused node
    int first_unused_node = num_nodes_s;
    for ( int i = current_node+1; i < num_nodes_s; ++i ) {
        if ( edges_per_node[i] == 0 ) {
            first_unused_node = i;
            break;
        }
    }

    assert( current_node < num_nodes_s && current_edge > 0 );

    if ( current_node == num_nodes_s - 1 ) {
        // we are done
        bool sa = screen_automorphism( );
        if ( sa ) {
            if ( check_2_connected( num_nodes_s,
                                    graph,
                                    edges_per_node  ) ) {
                (*callback_s)( num_nodes_s,
                               graph,
                               edges_per_node  );
            }
        }
        return;
    }


    // try connecting to previously unconnected node.
    if ( first_unused_node < num_nodes_s ) {
        // OK, found new node to add
        add_edge( current_node, current_edge, first_unused_node, 0 );
        add_edge_recursive( current_node, current_edge+1 );
        remove_edge( current_node, current_edge, first_unused_node, 0 );
    }

    // try advancing to next edge on current node or next node
    if ( current_edge < edges_per_node[ current_node ] ) {
        add_edge_recursive( current_node, current_edge+1 );
    } else if ( edges_per_node[ current_node ] >= 3 ) {
        add_edge_recursive( current_node+1, 1 );
    }


    // finally try adding new edge to existing node
    if ( current_edge > 0 )  {
        int next_node;
        int next_edge;
        // loop through all edges around the adjoining face.
        for( 
                find_next_edge_on_face( current_node, current_edge, &next_node, &next_edge );
                current_node != next_node || current_edge != next_edge ;
                find_next_edge_on_face( next_node, next_edge, &next_node, &next_edge ) ) {

           // edges are added only from lower number to higher number nodes.
           if ( next_node <= current_node )
              continue; 

           // check to insure there is no previous connection of these two nodes
           bool duplicate = false;
           for ( int j=0; j<edges_per_node[ next_node ]; ++j ) {
               if ( graph[ next_node ][ j ] == current_node ) {
                   duplicate = true;
                   break;
               }
           }

           if ( !duplicate ) {
               // OK to add this edge
               int next_edge_to_add = next_edge;
               // edges are listed in cyclical order, but keep the lowest numbered node first.
               if ( next_edge == 0 )
                   next_edge_to_add = edges_per_node[ next_node ];
               add_edge( current_node, current_edge, next_node, next_edge_to_add );
               add_edge_recursive( current_node, current_edge+1 );
               remove_edge( current_node, current_edge, next_node, next_edge_to_add );
           }
        }
    }
}



void add_edge( int node1, int edge1, int node2, int edge2 )
{
    // slide over existing edgees to make room
    for ( int i = edges_per_node[node1]; i > edge1; --i ) {
        graph[node1][i] = graph[node1][i-1];
    }
    graph[node1][edge1] = node2;
    ++edges_per_node[node1];

    for ( int i = edges_per_node[node2]; i > edge2; --i ) {
        graph[node2][i] = graph[node2][i-1];
    }
    graph[node2][edge2] = node1;
    ++edges_per_node[node2];
}


void remove_edge( int node1, int edge1, int node2, int edge2 )
{
    // slide over edges to make room
    --edges_per_node[node1];
    for ( int i = edge1; i < edges_per_node[node1]; ++i ) {
        graph[node1][i] = graph[node1][i+1];
    }

    --edges_per_node[node2];
    for ( int i = edge2; i < edges_per_node[node2]; ++i ) {
        graph[node2][i] = graph[node2][i+1];
    }
}


void find_next_edge_on_face( int current_node, int current_edge, int *next_node, int *next_edge )
{
    assert( current_edge > 0 );
    *next_node = graph[ current_node ][ current_edge - 1 ];

    // search next node for other end of edge
    for ( int i=0; i<edges_per_node[ *next_node ]; ++i ) {
        if ( graph[ *next_node ][ i ] == current_node ) {
            if ( i == 0 ) {
                *next_edge = edges_per_node[ *next_node ];
            } else {
                *next_edge = i;
            }
            return;
        }
    }
    // Not good.  We didn't find the other end of the edge.
    assert(0);
}


bool screen_automorphism( )
{
    // returns false if graph should be rejected based on preferred automorphism
    for ( int i=0; i<num_nodes_s; ++i ) {
        // node 0 of isomorphism must have maximal edge count
        if ( edges_per_node[i] < edges_per_node[0] ) {
            continue;
        }
        int max_connected_node = 0;
        for ( int m = 0; m < edges_per_node[i]; ++m ) {
            int cnc = edges_per_node[ graph[ i ][ m ] ];
            if ( cnc > max_connected_node ) {
                max_connected_node = cnc;
            }
        }
        if ( max_connected_node > edges_per_node[1] )  {
            // fail test.  Found a preferred isomorphism.
            return false;
        } else if ( max_connected_node < edges_per_node[1] ) {
            continue;
        }
        for ( int j=0; j < edges_per_node[i]; ++j ) {
            // node 1 of isomorphm must have maximal edge count
            // of all nodes connecting to node 0.
            if ( edges_per_node[ graph[i][j] ] < max_connected_node ) {
                continue;
            }
            for ( int reflect = 1; reflect > -2; reflect-=2 ) {
                if ( !sup_refl_s && reflect == -1 ) {
                    // don't check reflections unless we want to suppress them.
                    break;
                }
                // build node translation table for original node number to iso node numbering.
                int i2o_node[ SIZE ];
                int o2i_node[ SIZE ];
                for ( int x=0; x<num_nodes_s; ++x ) {
                    i2o_node[x] = SIZE;
                    o2i_node[x] = SIZE;
                }
                i2o_node[ 0 ] = i;
                i2o_node[ 1 ] = graph[i][j];
                o2i_node[ i ] = 0;
                o2i_node[ graph[i][j] ] = 1;
                int allocated_nodes = 2;
                for ( int a = 0; a < num_nodes_s; ++a ) {
                    assert( i2o_node[a] < SIZE );
                    // a is node number of iso graph
                    // i20_node[a] is corresponding node of original graph

                    // comparison resolved if different number of connections.
                    if ( edges_per_node[a] < edges_per_node[ i2o_node[a] ] ) {
                        // this would be a preferred isomorphism
                        return false;
                    } else if ( edges_per_node[a] > edges_per_node[ i2o_node[a] ] ) {
                        // current graph is preferred to this isomorphism,
                        // so move on to checking the next one.
                        goto next_iso;
                    }
                    int min_edge;
                    int min_node = SIZE;
                    for ( int b = 0; b<edges_per_node[ i2o_node[ a ] ]; ++b ) {
                        int onode = graph[ i2o_node[ a ] ][ b ];
                        int inode = o2i_node [ onode ];
                        if ( inode < min_node ) {
                            min_node = inode;
                            min_edge = b;
                        }
                    }
                    assert( min_node < SIZE );
                    // loop through all edges cyclically starting at min_edge
                    int bi = min_edge;
                    int bo = 0;
                    while(1) {
                        // find onode
                        int onode = graph[ i2o_node[ a ] ][ bi ];
                        if ( o2i_node[ onode ] == SIZE ) {
                            assert( allocated_nodes < num_nodes_s );
                            i2o_node[ allocated_nodes ] = onode;
                            o2i_node[ onode ] = allocated_nodes++;
                        }
                        int inode = o2i_node[ onode ];
                        int oonode = graph[ a ][ bo ];
                        if ( inode > oonode ) {
                            return false;
                        } else if ( inode < oonode ) {
                            goto next_iso;
                        }

                        bo++;
                        bi += reflect;
                        if ( bi == -1 )
                            bi = edges_per_node[ i2o_node[ a ] ] - 1;
                        if ( bi == edges_per_node[ i2o_node[ a ] ] )
                            bi = 0;
                        if ( bi == min_edge )
                            break;
                    }
                }
                next_iso:;
            }
        }
    }
    return true;
}


// This function is for debugging only.  It should return the same value as screen_automorphisms(),
// however it prints out all automorphisms in text format and does a string compare to select the
// preferred one.  This is much slower, but easier to inspect and understand.
bool search_automorphisms( )
{
    char original_string[ SIZE*2 ];
    char new_string[ SIZE*2 ];
    char max_string[ SIZE*2 ];
    char auto_list[ SIZE ][ SIZE*2 ];
    int auto_list_count = 0;
    char* op = original_string;
    char* np = new_string;
    memset( max_string, 0, sizeof( max_string ) );
    // printf("search automorphisms:\n");
    // print in compact form
    for ( int i=0; i<num_nodes_s; ++i ) {
        int x = edges_per_node[i];
        *op++ = x < 10 ? '0' + x : 'a' + x - 10;
        *op++ = '.';
        for ( int j=0; j < edges_per_node[i]; ++j ) {
            int x = graph[i][j];
            *op++ = x < 10 ? '0' + x : 'a' + x - 10;
        }
        *op++ = ':';
    }
    *op++ = 0;

    for ( int i=0; i<num_nodes_s; ++i ) {
        // node 0 of isomorphism must have maximal edge count
        if ( edges_per_node[i] < edges_per_node[0] ) {
            continue;
        }
        int max_connected_node = 0;
        for ( int m = 0; m < edges_per_node[i]; ++m ) {
            int cnc = edges_per_node[ graph[ i ][ m ] ];
            if ( cnc > max_connected_node ) {
                max_connected_node = cnc;
            }
        }
        for ( int j=0; j < edges_per_node[i]; ++j ) {
            // node 1 of isomorphm must have maximal edge count
            // of all nodes connecting to node 0.
            if ( edges_per_node[ graph[i][j] ] < max_connected_node ) {
                continue;
            }
            for ( int reflect = 1; reflect > -2; reflect-=2 ) {
                if ( !sup_refl_s && reflect == -1 ) {
                    // don't check reflections unless we want to suppress them.
                    break;
                }
                np = new_string;
                int i2o_node[ SIZE ];
                int o2i_node[ SIZE ];
                for ( int x=0; x<num_nodes_s; ++x ) {
                    i2o_node[x] = SIZE;
                    o2i_node[x] = SIZE;
                }
                i2o_node[ 0 ] = i;
                i2o_node[ 1 ] = graph[i][j];
                o2i_node[ i ] = 0;
                o2i_node[ graph[i][j] ] = 1;
                int allocated_nodes = 2;
                for ( int a = 0; a < num_nodes_s; ++a ) {
                    assert( i2o_node[a] < SIZE );
                    int x = edges_per_node[ i2o_node[ a ] ];
                    *np++ = x < 10 ? '0' + x : 'a' + x - 10;
                    *np++ = '.';
                    // a is node number of iso graph
                    // i20_node[a] is corresponding node of original graph
                    int min_edge;
                    int min_node = SIZE;
                    for ( int b = 0; b<edges_per_node[ i2o_node[ a ] ]; ++b ) {
                        int onode = graph[ i2o_node[ a ] ][ b ];
                        int inode = o2i_node [ onode ];
                        if ( inode < min_node ) {
                            min_node = inode;
                            min_edge = b;
                        }
                    }
                    assert( min_node < SIZE );
                    // loop through all edges cyclically starting at min_edge
                    int b = min_edge;
                    while(1) {
                        // find onode
                        int onode = graph[ i2o_node[ a ] ][ b ];
                        if ( o2i_node[ onode ] == SIZE ) {
                            assert( allocated_nodes < num_nodes_s );
                            i2o_node[ allocated_nodes ] = onode;
                            o2i_node[ onode ] = allocated_nodes++;
                        }

                        // print inode
                        int inode = o2i_node[ onode ];
                        assert( inode < num_nodes_s && inode >= 0 );
                        *np++ = inode < 10 ? '0' + inode : 'a' + inode - 10;

                        b += reflect;
                        if ( b == -1 )
                            b = edges_per_node[ i2o_node[ a ] ] - 1;
                        if ( b == edges_per_node[ i2o_node[ a ] ] )
                            b = 0;
                        if ( b == min_edge )
                            break;
                    }
                    *np++ = ':';
                }
                *np++ = 0;
                if ( strcmp( new_string, max_string ) > 0 )  {
                    strcpy( max_string, new_string );
                }
                // see if new string is in list
                bool found = false;
                for ( int k = 0; k < auto_list_count; ++k ) {
                    if ( strcmp( auto_list[k], new_string ) == 0 ) {
                        found = true;
                        break;
                    }
                }
                if ( !found ) {
                    strcpy( auto_list[ auto_list_count++ ], new_string );
                    // printf("     %d  %d %d  (%s)\n", i, j, reflect, new_string );
                }
            }
        }
    }
    printf( "%s", original_string );
    if ( strcmp( max_string, original_string ) > 0 ) {
        printf("  ---->  %s\n", max_string );
        return false;
    } else {
        printf("  MAX\n" );
        return true;
    }

}




void print_graph( )
{
    printf("GGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGG");
    for (int i=0; i<num_nodes_s; ++i ) {
        printf("\n%3d:", i );
        for (int j = 0; j<edges_per_node[i]; ++j ) {
            printf(" %3d", graph[i][j] );
        }
    }
    printf("\n\n");
}


bool check_2_connected( int num_nodes,
                        int graph[SIZE][SIZE],
                        int edges_per_node[SIZE]  )
{
    // the simplest way to check if 2-connected is
    // to check each face and see if the same node appears 
    // twice.
    bool marked[SIZE][SIZE];
    for ( int i = 0; i < num_nodes; ++i ) {
        for ( int j = 0; j < num_nodes; ++ j ) {
            marked[i][j] = false;
        }
    }
    for ( int i = 0; i < num_nodes; ++i ) {
        for ( int j = 0; j < edges_per_node[i]; ++ j ) {
            // check the face
            int node_marked[SIZE];
            int current_node = i;
            int current_edge = ( j == 0 ) ? edges_per_node[i] : j;
            int next_node;
            int next_edge;
            if ( !marked[current_node][current_edge] ) {
                memset( node_marked, 0 , num_nodes*sizeof( int ) );
                node_marked[i] = true;
                for( 
                    find_next_edge_on_face( current_node, current_edge, &next_node, &next_edge );
                    current_node != next_node || current_edge != next_edge ;
                    find_next_edge_on_face( next_node, next_edge, &next_node, &next_edge ) ) {

                    marked[next_node][next_edge] = true;
                    if ( node_marked[ next_node ] ) {
                        return false;
                    } else {
                        node_marked[next_node] = true;
                    }

                }
            }
        }
    }
    return true;
}



void graph_callback( int   num_nodes,
                     int   graph[SIZE][SIZE],
                     int   edges_per_node[SIZE]  ) {
    // print out the graph
    printf( "%d ", num_nodes );
    for ( int i = 0; i<num_nodes; ++i ) {
        for ( int j = 0; j<edges_per_node[i]; ++j ) {
            putchar( 'a' + graph[i][j] );
        }
        if ( i < num_nodes-1 ) {
            putchar( ',' );
        }
    }
    putchar( '\n' );
}



main( int argc, char* argv[] ) 
{
    int edge_count;
    int node_count;
    assert( argc == 3 );
    sscanf( argv[1], "%d", &edge_count );
    sscanf( argv[2], "%d", &node_count );
    find_graphs( node_count, edge_count, true, &graph_callback );
}


