#include <stdio.h>
#include <math.h>
#include <assert.h>
#include <string.h>
#include "chol.h"

double m[100][100];

const int STRING_LIST_SIZE=10000;
char string_list[ STRING_LIST_SIZE ][ SIZE ];
int string_list_count;

// node 0 is ground and does not correspond to a matrix row/column
// also matrix is symetric, so only the lower triangle is used.
// This function allows access to the matrix m in a way that
// hides both these anomalies and simplifies the accessing code.
double& select( int x, int y ) {
    // returns the matrix element associated with nodes x and y
    static double zero;
    if ( x == 0 || y == 0 ) {
        zero = 0.0;
        return zero;
    } else if ( x > y ) {
        return m[x-1][y-1];
    } else {
        return m[y-1][x-1];
    }
}

// networks are represented by string in the following format:
//      "bcdef,afec,abed,ace,adcbf,aeb"
// each node is represented by a group of letters in order (a,b,c,etc.)
// each group of letters lists the connecting nodes in clockwise order

void reorigin_network( int num_nodes, char* string, int new_a, int new_b )
{
    // this function takes a network and relabels node new_a and new_b as
    // a and be respectively.  The resulting network is returned in string
    // in a unique format determined by the following rules:
    // 1. for each node the lowest connecting node is listed first with
    //    other following in clockwise order.
    // 2. other than 'a', all other nodes first appear in the string in
    //    alphabetical order.

    char input[SIZE];
    strcpy( input, string );

    // create mappaing arrays from old node label to new
    int old2new[SIZE];
    int new2old[SIZE];

    // set unmapped nodes to num_nodes to make it easy to find minimum mapped.
    for ( int i = 0; i < num_nodes; ++i ) {
        old2new[i] = num_nodes;
    }

    // initialize the first two nodes.
    old2new[ new_a ] = 0;
    old2new[ new_b ] = 1;
    new2old[ 0 ] = new_a;
    new2old[ 1 ] = new_b;
    int number_mapped = 2;

    char* out_p = string;
    for ( int new_node=0; new_node<num_nodes; ++new_node ) {
        int old_node = new2old[ new_node ];
        assert( old_node < num_nodes );

        // find old node in input 
        char* old_node_p = input;
        for ( int j=0; j<old_node; ++j ) {
            while ( *old_node_p++ != ',' ) {
                assert( *old_node_p != 0 );
            }
        }

        // find the minimum connected node
        char* min_connected_node_p = 0;
        char* q = old_node_p;
        int min_connected_node = num_nodes;
        for( char c = *q; c != ',' && c != 0; c = *++q ) {
            if ( old2new[ c - 'a' ] < min_connected_node ) {
                min_connected_node = old2new[ c - 'a' ];
                min_connected_node_p = q;
            }
        }
        assert( min_connected_node_p != 0 );

        // loop through all connected nodes in cyclical order
        // starting with the min.
        q = min_connected_node_p;
        do {
            int old_connected_node = *q - 'a';

            // if this node is not yet mapped, map it now.
            if ( old2new[ old_connected_node ] == num_nodes ) {
                old2new[ old_connected_node ] = number_mapped;
                new2old[ number_mapped ] = old_connected_node;
                ++number_mapped;
                assert( number_mapped <= num_nodes );
            }

            // output the mapped node to return string
            *out_p++ = old2new[ old_connected_node ] + 'a';

            // advance to next node in cyclical order
            ++q;
            if ( *q == ',' || *q == 0 ) {
                // wrap back to beginning
                q = old_node_p;
            }

        }  while ( q != min_connected_node_p );

        *out_p++ = ',';
    }
    // overwrite final ',' with null terminator.
    out_p[ -1 ] = 0;
}


void strip_reference_node( char* string ) 
{
    // when exporting an N terminal network, node 0 is a reference node,
    // node 1 is ground, and nodes 2 - N+1 are the terminals.
    // for output, though, remove reference node and slide up other nodes by one.
    char* in = string;
    char* out = string;

    // skip over first node
    while ( *in++ != ',' ) {
        assert( *in != 0 );
    }

    // now copy rest of string, delete 'a' and decrement all others.
    char c;
    do {
        c = *in++;
        if ( c == 'a' ) {
            continue;
        }
        if ( c >= 'b' && c <= 'z' ) {
            c--;
        }
        *out++ = c;
    } while(c);
}

void add_node_between( int num_nodes, char* string, int node1, int node2 ) 
{
    // assumes there is a connection between node1 and node2.
    // an additional node is created, the connection:
    //     node1 -- node2
    // becomes
    //     node1 -- node3 -- node2
    // where node3 is the new one.

    char* p = string;
    int node = 0;

    // find node1 and node2.  Reconnected them to new node.
    do {
        if ( *p == ',' ) {
            ++node;
        }
        if ( node == node1 && *p - 'a' == node2 ) {
            *p = 'a' + num_nodes;
        }
        if ( node == node2 && *p - 'a' == node1 ) {
            *p = 'a' + num_nodes;
        }
    } while ( *++p );

    // add new node to end of string
    *p++ = ',';
    *p++ = 'a' + node1;
    *p++ = 'a' + node2;
    *p++ = 0;

    // now reorigin network to put in standard form.
    // nodes 0 and 1 stay the same.
    reorigin_network( num_nodes+1, string, 0, 1 );
}

void remove_edge_between( char* string, int node1, int node2 ) 
{
    // assumes there is a connection between node1 and node2.
    char* p = string;
    char* q = string;
    int node = 0;

    // find node1 and node2.  Remove connection.
    do {
        if ( *p == ',' ) {
            ++node;
        }
        if ( node == node1 && *p - 'a' == node2 ) {
            ++p;
            continue;
        }
        if ( node == node2 && *p - 'a' == node1 ) {
            ++p;
            continue;
        }
        *q++ = *p++;

    } while ( *p );
    *q = 0;
}

int num_edges_to_node( char* string, int node )
{
    int current_node = 0;
    int num_edges = 0;
    char* p = string;
    do {
        if ( *p == ',' ) {
            ++current_node;
        } else if ( current_node == node ) {
            ++num_edges;
        }
    } while ( *++p && current_node <= node );
    return num_edges;
}


bool nodes_connect( char* string, int node1, int node2 )
{
    // check if node1 connects to node2;
    int node = 0;
    char* p = string;
    do {
        if ( node == node1 && *p - 'a' == node2 ) {
            return true;
        }
        if ( *p == ',' ) {
            ++node;
        }
    } while ( *++p );
    return false;
}

void reflect_network( int num_nodes, char* string )
{
    // reverse the cyclical order of edges around each node.
    char* p = string;
    while(1) {
        char* q = p;

        // advance q to end node
        while ( *q != ',' && *q ) {
            ++q;
        }

        // now reverse character order
        while ( q > p ) {
            char temp = *--q;
            *q = *p;
            *p++ = temp;
        }

        // advance p to next node
        while ( *p != ',' ) {
            if ( *p == 0 ) {
                goto done;
            }
            ++p;
        }
        ++p;
    }
done:
    reorigin_network( num_nodes, string, 0, 1 );
}



int main( int argc, char* argv[] ) 
{

    int max_order = 1000;
    if ( argc == 2 ) {
        sscanf( argv[1], "%d", &max_order );
    }

    char input_string[ 1000 ];
    int num_nodes;
    //int squares[100][2];
    int order;
    for (;;) {
        int x = scanf("%d %s\n", &num_nodes, input_string );
        if ( x != 2 ) {
            return 0;
        }

        // start new list for isomrophism filtering.
        string_list_count = 0;

        int count[SIZE];
        int connection[SIZE][SIZE];
        int n = 0;
        count[0] = 0;

        // read graph string into connection matrix.
        order = 0;
        for ( char* p = input_string; *p; ++p ) {
            if ( *p == ',' ) {
                count[ ++n ] = 0;
                continue;
            }
            int x = *p - 'a';
            connection[ n ][ count[n]++ ] = x;
            ++order;
        }
        order = order/2;
        num_nodes = n+1;

        if ( order > max_order+6 )
            // input graph is too big.  Ignore it.
            continue;

        // each node with 4 connections may be used as reference node.
        // The four connecting refnodes are in cyclical order: ground, T1, T2, and T3
        for ( int refnode = 0; refnode < num_nodes; ++refnode ) {
            if ( count[refnode] != 4 ) {
                continue;
            }
            for (int ground_index = 0; ground_index < 4; ++ground_index ) {
                int ground_node = connection[ refnode ][ ground_index ];
                if ( order > max_order+4 ) {
                    // this graph can only be used for M-networks, not F-networks.
                    // do a quick screen to see if M-network possible.
                    int t0_node = connection[ refnode ][ ( ground_index + 1 ) % 4 ];
                    int t1_node = connection[ refnode ][ ( ground_index + 2 ) % 4 ];
                    int t2_node = connection[ refnode ][ ( ground_index + 3 ) % 4 ];
                    if (   count[ t1_node ] < 3
                         || !  nodes_connect( input_string, t0_node, t1_node  )
                         || !  nodes_connect( input_string, t1_node, t2_node  )
                         ||    nodes_connect( input_string, t0_node, t2_node  ) ) {
                        // This can't be used for M-network.
                        continue;
                    }
                }
                // try both reflections and use the preferred one.
                char preferred_string[ SIZE ];
                char reflected_string[SIZE];
                strcpy( preferred_string, input_string );
                reorigin_network( num_nodes, preferred_string, refnode, ground_node );
                strcpy( reflected_string, preferred_string );
                reflect_network( num_nodes, reflected_string );
                if ( strcmp( reflected_string, preferred_string ) > 0 ) {
                    // use the reflected string if lexically greater than unreflected one.
                    strcpy( preferred_string, reflected_string );
                }

                // check if preferred string is already in string list.
                // if so, discard it and continue
                bool duplicate_string = false;
                for ( int i = 0; i < string_list_count; ++i ) {
                    if ( strcmp( preferred_string, string_list[i] ) == 0 ) {
                        duplicate_string = true;
                        break;
                    }
                }
                if ( duplicate_string ) {
                    continue;
                }
                strcpy( string_list[ string_list_count++ ], preferred_string );

                for ( int sex = 0; sex < 2; ++sex ) {
                    // if sex is 1 (M) modify network accordingly
                    if ( sex == 1 ) {
                        if (    num_edges_to_node( preferred_string, 3 ) < 3
                             || !  nodes_connect( preferred_string, 2, 3 )
                             || !  nodes_connect( preferred_string, 3, 4 )
                             ||    nodes_connect( preferred_string, 2, 4 ) ) {
                            // not suitable for M network.
                            continue;
                        }
                        // make it an M-network.  
                        remove_edge_between( preferred_string, 2, 3 );
                        remove_edge_between( preferred_string, 3, 4 );
                    } else if ( order > max_order+4 ) {
                        // too big for F-network
                        continue;
                    }


                    // set up network matrix
                    // note that niether the reference node or the ground node gets
                    // a row/column in the matrix, so its dimension is num_nodes-2
                    for ( int i = 0; i< num_nodes-2; ++i ) {
                        for ( int j = 0; j <= i; ++j ) {
                            m[i][j] = 0.;
                        }
                    }
                    int node = 0;
                    for ( char* p = preferred_string; *p; p++ ) {
                        if ( *p == ',' ) {
                            ++node;
                        } else {
                            int connecting_node = *p - 'a';
                            if ( node == 0 || connecting_node == 0 || node > connecting_node ) {
                                // ignore reference node here
                                // only handle each edge once even though listed twice.
                                continue;
                            }
                            select( node-1, node-1 ) += 1.0;
                            select( connecting_node-1, connecting_node-1 ) += 1.0;
                            select( node-1, connecting_node-1 ) = -1.0;
                        }
                    }

                    // invert matrix to get resistance matrix
                    // use inverse_frac.  Assumes original matrix has only integer values.
                    // returns inverse a fractions with integer numerator stored in matrix
                    // and common denominator as return value;
                    double denominator = inverse_frac( m, num_nodes - 2 );

                    // Network may become disconnected resulting in singular matrix.
                    // Just continue in this case.
                    if ( denominator == 0 ) {
                        continue;
                    }

                    // screen network.  If any resistor voltages are identically zero or
                    // identically equal to another resistor voltage, then discard this network.
                    // compute the voltage across each resistor as a multiple of current injected
                    // into the terminals
                    double voltage[ SIZE ][ 3 ];
                    int rcount = 0;
                    node = 0;
                    for ( char* p = preferred_string; *p; p++ ) {
                        if ( *p == ',' ) {
                            ++node;
                        } else {
                            int connecting_node = *p - 'a';
                            if( node == 0 || connecting_node == 0 ) {
                                continue;
                            }
                            if ( connecting_node > node ) {
                                // each resister appears twice.  Just compute once.
                                for ( int terminal = 0; terminal < 3; ++terminal ) {
                                    voltage[ rcount ][ terminal ] = select( node-1, terminal + 1 )
                                                          - select( connecting_node-1, terminal + 1 );
                                }
                                ++rcount;
                            }
                        }
                    }

                    // now check resistance values
                    for ( int i = 0; i < rcount; ++i ) {
                        bool zero = true;
                        for ( int t = 0; t < 3; ++t ) {
                            if ( voltage[i][t] != 0 ) {
                                zero = false;
                                break;
                            }
                        }
                        if ( zero ) {
                            goto failed;
                        }
                        for ( int j = 0; j<i; ++j ) {
                            bool equal = true;
                            for ( int t = 0; t < 3; ++t ) {
                                if ( voltage[i][t] != voltage[j][t] ) {
                                    equal = false;
                                    break;
                                }
                            }
                            if ( equal ) {
                                goto failed;
                            }
                        }
                    }


                    char output_string[SIZE];
                    strcpy( output_string, preferred_string );


                    // now strip the reference node
                    strip_reference_node( output_string );

                    // output order, sex, resistance matrix, and network string.
                    printf( "%d ", sex ? order-6 : order-4 );
                    // number of terminals
                    printf( "%d ", 3 );
                    printf( sex ? "M " : "F " );
                    double rr[SIZE][SIZE];
                    for ( int t1 = 0; t1 < 3; ++t1 ) {
                        for ( int t2 = 0; t2 <= t1; ++t2 ) {
                            printf( "%g,", m[t1][t2] );
                            rr[t1][t2] = m[t1][t2] / denominator;
                        }
                    }
                    // print the denominator last
                    printf( "%g ", denominator );

                    // resistance metrics.
                    double rmax = ( rr[0][0] + rr[1][1] + rr[2][2] + 2*(rr[1][0] + rr[2][0] + rr[2][1] ) ) / 9. ;
                    inverse( rr, 3 );
                    double rmin = 1. / (rr[0][0] + rr[1][1] + rr[2][2] + 2*(rr[1][0] + rr[2][0] + rr[2][1] ) );
                    int irmin = int ( floor( rmin * ( 1 << 24 ) - 1e-7 ) );
                    int irmax = int( ceil( rmax * ( 1 << 24 )  + 1e-7 ) );
                    printf( "  %d %d  ", irmin, irmax );

                    // print the number of nodes
                    printf("%d ", num_nodes-1 );

                    // print the network string
                    printf( "%s\n", output_string );
                }
failed:;
            }
        }
    }
}




            


