#include "chol.h"
#include <math.h>
#include <assert.h>
#include <stdio.h>


double chol( double a[SIZE][SIZE], int n ) 
{
    // compute in place cholesky decompostion of a.
    // L is lower triangular and A = L * tr(L) 
    double det = 1.;
    for ( int j = 0; j< n; ++j ) {

        // compute r[j][j]
        double s = 0.;
        for ( int k = 0; k<j; ++k ) {
            double x = a[j][k];
            s = s + x*x;
        }
        s = a[j][j] - s;
        det *= s;
        a[j][j] = sqrt( s );

        for ( int i = j+1; i < n; ++i ) {
            // compute r[i][j]
            double s = 0.;
            for ( int k = 0; k < j; ++k ) {
                s += a[i][k] * a[j][k];
            }
            a[i][j] = ( a[i][j] - s ) / a[j][j];
        }
    }
    return det;
    //printf( "det = %g\n", det );
}

void inverse_chol( double c[SIZE][SIZE], int n ) {
    // compute the inverse of the cholesky matrix
    // c is cholesky matrix.
    
    for ( int i = 0; i<n; ++i ) {
        c[i][i] = 1./c[i][i];
        for ( int j = 0; j < i; ++j ) {
            double s = 0.;
            for ( int k = j; k < i; ++k ) {
                s += c[i][k]*c[k][j];
            }
            c[i][j] = -s*c[i][i];
        }
    }
}

double inverse( double c[SIZE][SIZE], int n ) {
    // compute inverse of SPD matrix. Only lower triangle is used
    double det = chol( c, n );
    if ( det == 0. ) {
        return 0.;
    }
    inverse_chol( c, n );
    // now compute Ct*C
    for ( int j = 0; j < n; ++j ) {
        for ( int i = j; i < n; ++i ) {
            double sum = 0.;
            for ( int k = i; k < n; ++ k ) {
                sum += c[k][i] * c[k][j];
            }
            c[i][j] = sum;
        }
    }
    return det;
}

double gcd( double x, double y )
{
    assert( x==floor(x) && y==floor(y) );
    if ( y==0 ) {
        return x;
    }
    double q = floor( x / y );
    double r = x - q * y;
    assert( fabs(r) < fabs(y) );
    if ( r==0. ) {
        // good enough
        return fabs(y);
    } else {
        return gcd( y, r );
    }
}


double inverse_frac( double m[SIZE][SIZE], int n ) 
{
    // assumes m has integer components.
    // returnes inverse as fractions with numerators in m
    // and common denominator as return value.
    double det = inverse( m, n );
    double idet = floor( det + 0.5 );
    if ( !isnormal( idet ) ) {
        return 0.;
    }
    double d = idet;
    for ( int i = 0; i < n; ++i ) {
        for ( int j = 0; j <= i; ++j ) {
            double x = idet*m[i][j];
            assert( fabs(x) < 1e15 );
            x = floor( x + 0.5 );
            m[i][j] = x;
            d = gcd( d, x );
        }
    }
    idet /= d;
    for ( int i = 0; i < n; ++i ) {
        for ( int j = 0; j <= i; ++j ) {
            m[i][j] /= d;
        }
    }
    return idet;
}



void print_matrix( double m[SIZE][SIZE], int n )
{
    for ( int i=0; i < n; ++i ) {
        for ( int j=0; j < n; ++j ) {
            printf( "%15g  ", m[i][j] );
        }
        printf( "\n" );
    }
    printf( "\n" );
}

