#include "Graph.h"
#include "Matrix.h"
#include "Primes.h"
#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <unistd.h>
#include <math.h>
#include <string.h>


u32 max_edges;
u32 max_nodes;
u32 mod_count;
u32 modulus;
u32 mod_threshold;
double min_square_side;
u32 connectivity;
bool filter;
bool filter_squares;
bool filter_aps;

double primes[] = { PRIMES };

double max_square_factor( double x ) {
    u32 i, j;
    double square_factor = 1.;
    u32 num_primes = sizeof(primes)/sizeof(double);
    for ( i = 0; ; ++i ) {
        assert( i < num_primes ); // add more primes to the list if this ever fails.
        double p = primes[i];
        for ( j = 0; j < 100; ++j ) {
            double q = x/p;
            if ( q == round(q) ) {
                x = q;
                if ( j & 1 ) {
                    square_factor *= p;
                }
            } else {
                break;
            }
        }
        if ( ( i & 7 ) == 0 ) {
            // every 8 times through the loop, see if we can quit.
            if ( p*p*p >= x )
                // x has no prime factors <= cube_root(x).
                break;
        }
    }
    double y = round( sqrt( x ) );
    if ( y*y == x )
        return y*square_factor;
    else
        return square_factor;
}

bool check_determinant( double d, u32 num_edges ) 
{
    if ( filter_squares ) {
        double d2 = d/2.;
        if ( d2 != round(d2) )
            return false;
        return ( max_square_factor( d2 ) >= min_square_side );
    } else if ( filter_aps ) {
        if ( num_edges == max_edges ) {
            double d13 = d/13.;
            return ( d13 == round(d13) );
        } else if ( num_edges == max_edges-1 ) {
            double d13 = d/13.;
            double d21 = d/21.;
            return ( d13 == round(d13) || d21 == round(d21) );
        } else {
            return true;
        }
    }
    assert(0);
}


void output_graph( Graph &g, char* string )
{
    printf( "%d %s\n", g.num_nodes, string );
}

bool mod_check()
{
    bool ret_value = false;
    if ( ++mod_count >= modulus ) {
        ret_value = true;
        mod_count = 0;
    }
    return ret_value;
}

// If this graph was previously just extended by an OPEN extension,
// then ndeg0 and ndeg1 are the larger and smaller degrees, respectively,
// of the two nodes connected by the edge just added.  Any new edge
// added must beat out this edge to be the preferred reductions,
// so this allows some early screening of potential extensions.
// Set nedg0 and ndeg1 to zero if this graph was not created by an
// OPEN extension.
void extend_graph( Graph& g, Matrix* m, u32 ndeg0, u32 ndeg1 ) {

    // the preferred reduction algorithm guarantees that valid
    // extensions of non-isomorphic graphs won't end up being 
    // isomorphic.  However multiple extensions of the same graph
    // may well be isomorphic.  So we filter out these isomorphisms
    // using the StringSet class.
    StringSet string_set;
    string_set.reset();

    for ( u32 f=0; f < g.num_faces; ++f ) {
        u32 fdeg = g.face_degree[f];
        // OPEN extensions can only be done on a face of degree at least 4.
        if ( fdeg < 4 ) {
            continue;
        }
        DEdge* start_edge = g.face_edge[f];
        DEdge* ref0 = start_edge;
        do {
            u32 ref0_node_deg = g.node_degree[ ref0->node_num ];
            if ( ref0_node_deg + 1 >= ndeg0 ) {
                DEdge* ref1 = ref0->other_end->cc_next->other_end->cc_next;
                for ( u32 i = 0; i < fdeg - 3;++i ) {
                    u32 ref1_node_deg = g.node_degree[ ref1->node_num ];

                    // each potential extension edge can be added in two direction,
                    // so screen out one of these and just investigate the other
                    // to save time
                    if ( ref1_node_deg > ref0_node_deg )
                        goto next_ref1;
                    if ( ref0_node_deg + 1 == ndeg0 && ref1_node_deg + 1 < ndeg1 )
                        goto next_ref1;
                    if ( ref1_node_deg == ref0_node_deg  && ref0->node_num > ref1->node_num )
                        goto next_ref1;
                    // Make sure this edge is not a parallel to an existing edge.  For 3-connected graphs
                    // two non-sequential nodes on a single face will never connect, but for 2-connected
                    // graphs we need to check explicitly.
                    if ( connectivity == 2 && ref0->node_connects_to( ref1->node_num ) )
                        goto next_ref1;

                    {
                        u32 n0 = ref0->node_num;
                        u32 n1 = ref1->node_num;
                        bool det_ok = true;
                        if ( filter ) {
                            det_ok = check_determinant( m->determinant() + m->transfer_impedance( n0, n1, n0, n1 ), g.num_edges+1 );
                        }
                        // When we get to maximum edge count, quit early if determinant looks no good for square.
                        // At less than max count, we need to continue extending the graph.
                        if ( det_ok || g.num_edges < max_edges-1  ) {
                            g.add_open( ref0, ref1 );
                            Edge* new_edge = ref0->cl_next->parent;
                            OEdge* e = g.max_oedge( new_edge );
                            if ( g.check_preferred_reduction( e, OPEN, connectivity ) ) {
                                char * string = string_set.allocate_string();
                                e->print_graph( string );
                                if ( !string_set.check_and_include( string ) ) {
                                    // we have a valid graph
                                    bool mod_passes = true;
                                    if ( g.num_edges <= mod_threshold )
                                        mod_passes = mod_check();
                                    if ( mod_passes ) {
                                        if ( det_ok ) {
                                             output_graph( g, string );
                                        }
                                    }
                                    if ( g.num_edges < max_edges && ( mod_passes || g.num_edges < mod_threshold ) ) {
                                        if (filter) {
                                            Matrix m2(*m);
                                            m2.add_edge( n0, n1 );
                                            extend_graph( g, &m2, g.node_degree[ e->parent->node_num ], g.node_degree[ e->parent->other_end->node_num ] );
                                        } else {
                                            extend_graph( g, 0, g.node_degree[ e->parent->node_num ], g.node_degree[ e->parent->other_end->node_num ] );
                                        }
                                    }
                                }
                            }
                            g.remove_open( new_edge );
                        }
                        // we need to restore face_edge here, although it only matters for 
                        // res/mod ordering.  "remove_open" puts it back correctly, but 
                        // not identically which will affect the order (but not correctness)
                        // of future expansions.
                        g.face_edge[f] = start_edge;
                    }
                next_ref1:
                    ref1 = ref1->other_end->cc_next;
                }
            }
            ref0 = ref0->other_end->cc_next;
        } while (ref0 != start_edge );
    }

    // a SHORT extension or DIAMOND extension increases the number of nodes, so don't bother
    // checking if we are already at the maximum.
    if ( g.num_nodes >= max_nodes )
        return;

    // In order for a SHORT extension or DIAMOND extension to become
    // the preferred reduction, there must be no valid open reductions.
    // Find any open reductions in the current graph
    // and keep track of the edges and nodes.
    // If there are any open reductions, then any short
    // expansion must be based on a node that is shared by
    // all open reductions.  If there is exactly one open reduction
    // than any diamond expansion must use the same edge.  If there
    // is more than one elegable open reduction, then no diamond
    // expansion is possible.
    u32 node_a = ~0u;
    u32 node_b = ~0u;
    Edge* open_reduction_edge;
    bool open_reduction_found = false;
    bool two_open_reductions_found = false;

    for ( u32 n=0; n < g.num_nodes; ++n ) {
        u32 ndeg = g.node_degree[n];
        if ( ndeg < 4 ) {
            continue;
        }
        DEdge* d = g.node_edge[n];
        DEdge* dd = d;
        do {
            u32 node0 = d->node_num;
            u32 node1 = d->other_end->node_num;
            if ( node0 < node1 && g.node_degree[ node1 ] >= 4  ) {
                if ( connectivity == 3 ? g.check_open_3con(d->parent) : g.check_open_2con(d->parent) ) {
                    if ( !open_reduction_found ) {
                        open_reduction_found = true;
                        open_reduction_edge = d->parent;
                        node_a = node0;
                        node_b = node1;
                    } else {
                        two_open_reductions_found = true;
                        if ( node_a != node0 && node_a != node1 )
                            node_a = ~0u;
                        if ( node_b != node0 && node_b != node1 )
                            node_b = ~0u;
                        if ( node_a == ~0u && node_b == ~0u )
                            return;
                    }
                }
            }
            d = d->cl_next;
        } while ( d != dd );
    }
    bool one_node = open_reduction_found && ( node_a == ~0u || node_b == ~0u );
    if ( node_a == ~0u )
        node_a = node_b;

    // start searching for SHORT extensions
    string_set.reset();
    for ( u32 n=0; n < g.num_nodes; ++n ) {
        if ( one_node && n != node_a )
           continue;
        u32 ndeg = g.node_degree[n];
        // short extension must be done on node of degreee at least 4.
        if ( ndeg < 4 ) {
            continue;
        }
        DEdge* start_edge = g.node_edge[n];
        DEdge* ref0 = start_edge;
        do {
            DEdge* ref1 = ref0->cl_next->cl_next;
            for ( u32 i = 0; i < ndeg - 3;++i ) {
                g.add_short( ref0, ref1 );
                Edge* new_edge = ref0->cl_next->parent;
                OEdge* e = g.max_oedge( new_edge );
                if ( g.check_preferred_reduction( e, SHORT, connectivity ) ) {
                    char * string = string_set.allocate_string();
                    e->print_graph( string );
                    if ( !string_set.check_and_include( string ) ) {
                        // we have a valid graph
                        // recalculate m here - not the performance path
                        Matrix m2( g );
                        m2.invert();
                        bool mod_passes = true;
                        if ( g.num_edges <= mod_threshold )
                            mod_passes = mod_check();
                        if ( mod_passes ) {
                            output_graph( g, string );
                        }
                        if ( g.num_edges < max_edges && ( mod_passes || g.num_edges < mod_threshold ) ) {
                            extend_graph( g, &m2, 0, 0 );
                        }
                    }
                }
                g.remove_short( new_edge );
                g.node_edge[n] = start_edge;
                ref1 = ref1->cl_next;
            }
            ref0 = ref0->cl_next;
        } while (ref0 != start_edge );
    }

    // check for any diamond expansions
    string_set.reset();
    if ( !two_open_reductions_found && connectivity == 2 && g.num_edges <= max_edges - 4 && g.num_nodes <= max_nodes - 2 )  {
        // The above check will fail 99+% of the time, so the following is not very
        // performance critical.  Just keep it simple.
        for ( Edge* e = g.edge_list; e; e = e->next ) {
            // after adding and then removing a diamond, g.node_edge and g.face_edge may be 
            // different.  So we need to restore them to the original value. Keep it simple 
            // by just copying the whole array to a temporary space on the stack.
            u32 temp_node_edge[ MAX_GRAPH_SIZE ];
            u32 temp_face_edge[ MAX_GRAPH_SIZE ];
            memcpy( temp_node_edge, g.node_edge, g.num_nodes*sizeof(g.node_edge[0]) );
            memcpy( temp_face_edge, g.face_edge, g.num_faces*sizeof(g.face_edge[0]) );
            g.add_diamond( e );
            OEdge* moe = g.max_oedge( e );
            if ( g.check_preferred_reduction( moe, DIAMOND, connectivity ) ) {
                char * string = string_set.allocate_string();
                moe->print_graph( string );
                if ( !string_set.check_and_include( string ) ) {
                    // this is a valid expansion
                    Matrix m2( g );
                    m2.invert();
                    bool mod_passes = true;
                    // diamond extention adds 4 edges, so if previous extention was
                    // not pruned based on res/mod, then we need to do so here if
                    // pre-extension, the number of edges was less than mod_threshold.
                    if ( g.num_edges <= mod_threshold + 3 )
                        mod_passes = mod_check();
                    if ( mod_passes ) {
                        output_graph( g, string );
                    }
                    if ( g.num_edges < max_edges && ( mod_passes || g.num_edges < mod_threshold ) ) {
                        extend_graph( g, &m2, 0, 0 );
                    }
                }
            }
            g.remove_diamond( e );
            memcpy( g.node_edge, temp_node_edge, g.num_nodes*sizeof(g.node_edge[0]) );
            memcpy( g.face_edge, temp_face_edge, g.num_faces*sizeof(g.face_edge[0]) );
        }
    }
}



int main( int argc, char* argv[] ) {
    Graph g;
    int opt;
    mod_threshold = 25;
    min_square_side = 0.;
    while ( ( opt = getopt( argc, argv, "t:s:ae:n:r:m:c:" ) ) != -1 ) {
        if ( opt=='s' ) {
            filter_squares = true;
            filter = true;
            int status = sscanf( optarg, "%lf", &min_square_side );
            assert( status == 1 );
        } else if ( opt=='a' ) {
            filter_aps = true;
            filter= true;
        } else if ( opt=='t' ) {
            int status = sscanf( optarg, "%d", &mod_threshold );
            assert( status == 1 );
        } else if ( opt=='e' ) {
            int status = sscanf( optarg, "%d", &max_edges );
            assert( status == 1 );
        } else if ( opt == 'n' ) {
            int status = sscanf( optarg, "%d", &max_nodes );
            assert( status == 1 );
        } else if ( opt == 'r' ) {
            int status = sscanf( optarg, "%d", &mod_count );
            assert( status == 1 );
        } else if ( opt == 'm' ) {
            int status = sscanf( optarg, "%d", &modulus );
            assert( status == 1 );
        } else if ( opt == 'c' ) {
            int status = sscanf( optarg, "%d", &connectivity );
            assert( status == 1 );
            assert( connectivity == 2 || connectivity == 3 );
        } else {
            fprintf(stderr, "Unrecognized options\n" );
            assert(0);
        }
    }

    if ( max_nodes == 0 )
        max_nodes = (max_edges+2)/2;
    if ( connectivity == 0 ) {
        connectivity = 2;
    }

    for ( u32 wheel_size = 3; wheel_size <= max_edges/2 && wheel_size+1 <= max_nodes; ++wheel_size ) {
        g.create_wheel( wheel_size );

        char string[256];
        bool mod_passes = mod_check();
        if ( mod_passes ) {
            g.oedge_list->print_graph( string );
            output_graph(g, string);
        }
        if ( g.num_edges < max_edges  && ( mod_passes || g.num_edges < mod_threshold ) ) {
            Matrix m( g );
            m.invert();
            extend_graph(g, &m, 0, 0 );
        }
    }
}


