#include <stdio.h>
#include <math.h>
#include <assert.h>
#include <string.h>
#include <unistd.h>
#include "Matrix.h"
#include "Graph.h"
#include "Filter.h"

Filter filter;
Graph graph;

void usage() {
    fprintf(stderr, "Usage:  findpr [ -a <augmentation_limit> ] [ -o <max_cpss_order>  -f <filter_file> ][ -s ][ -d ]\n" );
    fprintf(stderr, "   -a <augmentation_limit>               // creates augmented rectangles up to order <augmentation linit>\n");
    fprintf(stderr, "   -o <max_cpss_order>  -f <filter_file> // applies filter file to select PRs for CPSS of up to order <max_cpss_order>\n");
    fprintf(stderr, "   -s                                    // print only perfect squares\n");
    fprintf(stderr, "   -d                                    // Print duals.  Normally not printed to avoid printing PRs/PSs twice\n");
    _exit(1);
}


// input is expected to be all planar graphs
int main( int argc, char *argv[] ) 
{
    // grab arguments
    int opt;
    int augment_limit = 0;
    int max_square_order = 0;
    bool print_squares_only = false;
    bool print_duals = false;
    char* filter_file = 0;
    while ( (opt = getopt( argc, argv, "a:o:f:sd" ) ) != -1 ) {
            if (opt=='a') {
                int status = sscanf( optarg, "%d", &augment_limit );
                if ( status != 1 )
                    usage();
            }
            else if ( opt=='o' ) {
                int status = sscanf( optarg, "%d", &max_square_order );
                if ( status != 1 )
                    usage();
            }
            else if ( opt=='f' ) {
                filter_file = optarg;
            }
            else if ( opt=='s' ) {
                print_squares_only = true;
            }
            else if ( opt=='d' ) {
                print_duals = true;
            }
            else {
                usage();
            }
    }

    if ( filter_file )
        filter.read_filter_data( filter_file );
    else if ( max_square_order ) 
        usage();

    for (;;) {
        int num_nodes;
        char string[128];
        int x = scanf("%d %s\n", &num_nodes, string );
        if ( x != 2 ) {
            return 0;
        }

        // create a graph object
        graph.create_from_string( string ); 

        // reject dual if num nodes is greater than num faces
        if ( !print_duals && graph.num_nodes > graph.num_faces ) 
            continue;

        // create resistance matrix
        Matrix m( graph );
        double determinant = m.invert();

        // loop through grpah edges
        for ( Edge* ref_edge = graph.edge_list; ref_edge; ref_edge=ref_edge->next ) {

            // top_node and bottom_node are the two nodes connected by the edge
            int top_node = ref_edge->dedge_0.node_num;
            int bottom_node = ref_edge->dedge_1.node_num;

            // check aspect ratio of rectangle
            double t = m.transfer_impedance( top_node, bottom_node, top_node, bottom_node );
            double tc = determinant - t;
            double aspect_ratio;
            if ( tc > t ) {
                aspect_ratio = tc/t;
            } else {
                aspect_ratio = t/tc;
            }
            if ( print_squares_only && t != tc ) {
                continue;
            }

            // filter on aspect ratio
            // do quick check for max sizes
            u32 current_order = graph.num_edges - 1;
            if ( max_square_order - current_order == 6 ) {
                if ( fabs( aspect_ratio - 8./5. ) > 1e-13 )
                    continue;
            } else if ( max_square_order - current_order == 7 ) {
                if (     fabs( aspect_ratio - 8./5. ) > 1e-13
                     &&  fabs( aspect_ratio - 13./8. ) > 1e-13 )
                    continue;
            } else if ( max_square_order > 0 )  {
                if ( !filter.check_aspect_ratio( aspect_ratio, max_square_order - current_order ) ) 
                    continue;
            }

            // now calculate node voltages.
            double voltage[64];
            for ( int j =0; j<num_nodes; ++j )  {
                voltage[j] =  m.select(j,top_node) - m.select(j,bottom_node);
            }

            // find greatest common divisor
            double g = 0;
            for ( int j =0; j<num_nodes; ++j )  {
                g = gcd( voltage[j], g );
            }

            // divide through by g
            for ( int j =0; j<num_nodes; ++j )  {
                voltage[j] = voltage[j]/g;
            }

            // compute square sizes
            struct Square_t {
                double size;
                double xpos;
                double ypos;
            } square[64];
            u32 num_squares = 0;
            for ( Edge* ee = graph.edge_list; ee; ee=ee->next ) {
                if ( ee == ref_edge ) 
                    continue;
                square[num_squares++].size = fabs( voltage[ ee->dedge_0.node_num ] - voltage[ ee->dedge_1.node_num ] );
            }

            // check if all different
            bool perfect = true;
            for ( u32 i = 0; i< num_squares; ++i ) {
                if ( square[i].size == 0. ) {
                    perfect = false;
                    break;
                }
                for ( u32 j = 0; j< i; ++j ) {
                    if ( square[i].size == square[j].size )  {
                        perfect = false;
                        break;
                    }
                }
                if ( !perfect ) {
                    break;
                }
            }

            if ( !perfect ) {
                continue;
            }

            // at this point we have a perfect rectangle.  Print it.
            //
            // compute "potential" for each face.  Each node of graph
            // represents horizontal line in PR and its voltage represents
            // its vertical position.  Each face of graph represents a vertical
            // line in the square and its potential represents its horizontal
            // position.

            double face_potential[64];
            bool done=false;
            u64 face_potential_computed = 0;
            u32 left_face = ref_edge->dedge_0.face_num;
            face_potential[left_face] = 0.;
            face_potential_computed |= 1uLL << left_face;

            while( !done ) {
                for ( Edge* ee = graph.edge_list; ee; ee=ee->next ) {
                    if ( ee == ref_edge ) 
                        continue;
                    u32 face0 = ee->dedge_0.face_num;
                    u32 face1 = ee->dedge_1.face_num;
                    u32 node0 = ee->dedge_0.node_num;
                    u32 node1 = ee->dedge_1.node_num;
                    if (   !( face_potential_computed & 1uLL << face0 )
                         && ( face_potential_computed & 1uLL << face1 ) ) {
                        // swap faces and nodes
                        u32 temp = face0;
                        face0 = face1;
                        face1 = temp;
                        temp = node0;
                        node0 = node1;
                        node1 = temp;
                    }
                    if (    ( face_potential_computed & 1uLL << face0 )
                         &&!( face_potential_computed & 1uLL << face1 ) ) {
                        face_potential[ face1 ] = face_potential[ face0 ]
                                                + voltage[ node1 ]
                                                - voltage[ node0 ];
                        face_potential_computed |= 1uLL << face1;
                        if ( face_potential_computed + 1 == 1uLL << graph.num_faces ) {
                            done = true;
                            break;
                        }
                    }
                }
            }

            // at this point we can compute the X/Y position of each component square within
            // the rectangle.
            double vbottom = voltage[ bottom_node ];
            {
                u32 i = 0;
                for ( Edge* ee = graph.edge_list; ee; ee=ee->next ) {
                    if ( ee == ref_edge ) 
                        continue;
                    u32 node0 = ee->dedge_0.node_num;
                    u32 node1 = ee->dedge_1.node_num;
                    u32 face0 = ee->dedge_0.face_num;
                    u32 face1 = ee->dedge_1.face_num;
                    square[i].ypos = fmin( voltage[node0], voltage[node1] ) - vbottom;
                    square[i].xpos = fmin( face_potential[face0], face_potential[face1] );
                    ++i;
                }
            }
            double total_height = voltage[ top_node ] - vbottom;
            double total_width = determinant/g - total_height;

            // put the rectangle in preferred orientation.  This is not 
            // always necessary, but makes testing a bit easier

            // find the largest corner
            enum { INVALID, LOWER_LEFT, UPPER_LEFT, LOWER_RIGHT, UPPER_RIGHT } max_corner = INVALID;
            double max_corner_size = 0.;
            for ( u32 i = 0; i < num_squares; ++i ) {
                double x = square[i].xpos;
                double y = square[i].ypos;
                double size = square[i].size;
                if ( size > max_corner_size ) {
                    if ( x == 0. && y == 0. ) {
                        max_corner = LOWER_LEFT;
                        max_corner_size = size;
                    } else if ( x == 0. && y+size == total_height ) {
                        max_corner = UPPER_LEFT;
                        max_corner_size = size;
                    } else if ( x+size == total_width && y == 0. ) {
                        max_corner = LOWER_RIGHT;
                        max_corner_size = size;
                    } else if ( x+size == total_width && y+size == total_height ) {
                        max_corner = UPPER_RIGHT;
                        max_corner_size = size;
                    }
                }
            }
            assert( max_corner_size > 0. && max_corner != INVALID );
            bool reflect_x = ( max_corner == LOWER_RIGHT || max_corner == UPPER_RIGHT );
            bool reflect_y = ( max_corner == UPPER_LEFT || max_corner == UPPER_RIGHT );

            // now put the max corner in the lower left
            for ( u32 i = 0; i < num_squares; ++i ) {
                if ( reflect_x ) 
                    square[i].xpos = total_width - square[i].xpos - square[i].size;
                if ( reflect_y ) 
                    square[i].ypos = total_height - square[i].ypos - square[i].size;
            }

            // now see if  we need to reflect across diagonal (swap height and width)
            // If not a square, then make width larger then height.  If a square, make
            // the square immediately to the right of the first be the larger of the two choices.
            bool swap_xy;
            if ( total_width == total_height ) {
                double size_first_right = 0.;
                double size_first_above = 0.;
                for ( u32 i = 0; i < num_squares; ++i ) {
                    if ( square[i].ypos==0. && square[i].xpos==max_corner_size )
                        size_first_right = square[i].size;
                    if ( square[i].xpos==0. && square[i].ypos==max_corner_size )
                        size_first_above = square[i].size;
                }
                assert( size_first_right > 0. && size_first_above > 0. );
                swap_xy = size_first_above > size_first_right;
            } else {
                swap_xy = ( total_height > total_width );
            }

            if ( swap_xy ) {
                if ( !print_duals && graph.num_nodes == graph.num_faces )
                    // in this case we want to skip this rectangle due to the dual rejection algorithm.
                    continue;
                double temp = total_width;
                total_width = total_height;
                total_height = temp;
                for ( u32 i = 0; i < num_squares; ++i ) {
                    temp = square[i].ypos;
                    square[i].ypos = square[i].xpos;
                    square[i].xpos = temp;
                }
            }
                
            // now sort the squares in preparation for printing.
            for ( u32 i = 1; i < num_squares; ++i ) {
                for ( u32 j = i; j >= 1; --j ) {
                    if ( square[j].ypos < square[j-1].ypos
                            || (    square[j].ypos == square[j-1].ypos
                                &&  square[j].xpos  < square[j-1].xpos  ) ) {
                        // swap
                        Square_t temp = square[j];
                        square[j] = square[j-1];
                        square[j-1] = temp;
                    } else {
                        break;
                    }
                }
            }

            // print the bouwkamp code to the string
            char output_string[1024];
            char* p = output_string;

            p += sprintf(p, "%d %0.0f %0.0f (", num_squares, total_width, total_height );
            for ( u32 i = 0; i < num_squares; ++i ) {
                 p += sprintf(p, "%0.0f", square[i].size );
                 // now print separator
                 if ( i == num_squares-1 ) {
                     p += sprintf(p, ")");
                 } else if ( square[i].ypos != square[i+1].ypos || square[i].xpos + square[i].size != square[i+1].xpos ) {
                     p += sprintf(p, ")(");
                 } else {
                     p += sprintf(p, ",");
                 }
            }
            // print the rectangle
            printf ("%s\n", output_string );

            // now print all augmented rectangles that were requested.
            for ( int i = 0; i < 2; ++i ) {
                // "i" indicates if augmenting width or height first
                char tstring[1024];
                char tstring2[1024];
                strcpy( tstring, output_string );
                u32 width = total_width;
                u32 height = total_height;
                for ( int j = num_squares+1; j <= augment_limit; ++j ) {
                    // alternately add full size square to width or height
                    int wh = ( i + j ) % 2;
                    if ( wh ) {
                        width = width + height;
                    } else {
                        height = width + height;
                    }
                    char *p = tstring2;
                    p += sprintf( tstring2, "%d %d %d ", j, width, height );
                    char *q = tstring;
                    // find first character after "("
                    while ( *q && *q != '(' )
                        ++q;
                    assert( *q );
                    ++q;
                    assert( *q );
                    if ( wh ) {
                        p += sprintf( p, "(%d,", height );
                    } else {
                        p += sprintf( p, "(%d)(", width );
                    }
                    strcpy( p, q );
                    strcpy( tstring, tstring2 );
                    printf( "%s\n", tstring );
                }
            }
        }
    }
}

