#include <stdio.h>
#include <math.h>
#include <assert.h>
#include <string.h>
#include <unistd.h>
#include "Filter.h"
#include "Matrix.h"
#include "Graph.h"


Filter filter;
Graph graph;

int main( int argc, char *argv[] ) 
{
    char string[ 1000 ];
    u32 num_nodes;

    // grab arguments
    int max_square_order = 0;
    char* filter_file = 0;
    int opt;
    while ( ( opt = getopt( argc, argv, "o:f:" ) ) != -1 ) {
        if ( opt=='o' ) {
            int status = sscanf( optarg, "%d", &max_square_order );
            assert( status == 1 );
        } else if ( opt == 'f' ) {
            filter_file = optarg;
        } else {
            fprintf(stderr, "Unrecognized options\n" );
            assert(0);
        }
    }

    assert( max_square_order == 0 || filter_file );
    if ( filter_file )
        filter.read_filter_data( filter_file );

    for (;;) {
        int x = scanf("%d %s\n", &num_nodes, string );
        if ( x != 2 ) {
            return( 0 );
        }

        // create graph and resistance matrix
        graph.create_from_string( string );
        // reject graphs with more nodes that faces, as we will
        // use its dual (which has num_nodes and num_faces exchanged).
        if ( graph.num_nodes > graph.num_faces ) 
            continue;
        assert( num_nodes == graph.num_nodes );
        Matrix m( graph );
        double determinant = m.invert();

        // loop through terminals
        for ( Edge* terminal_edge = graph.edge_list; terminal_edge; terminal_edge = terminal_edge->next ){
            int x = terminal_edge->dedge_0.node_num;
            int y = terminal_edge->dedge_1.node_num;

            // loop through rectangle select
            for ( Edge* rectangle_edge = graph.edge_list; rectangle_edge; rectangle_edge = rectangle_edge->next ){
                // ###############################################
                // this is the critical performance section of code
                // ###############################################
                if ( rectangle_edge == terminal_edge ) {
                    continue;
                }
                int rect_x = rectangle_edge->dedge_0.node_num;
                int rect_y = rectangle_edge->dedge_1.node_num;

                // inject current +/-ir into nodes rect_x and rect_y
                // inject current +/-1 into nodes x and y
                // solve for ir such that voltage x to y is 1/2.
                // 0.5 = transfer_impedance( x, y, x, y ) + ir*transfer_impedance( rect_x, rect_y, x, y )
                //
                // ir = ( 1 - 2*transfer_impedance( x, y, x, y ) ) / ( 2*transfer_impedance( rect_x, rect_y, x, y ) )

                // express IR as fraction. Transfer impedances are all scaled by determinant.
                double ir_numerator =  ( determinant - 2*m.transfer_impedance( x, y, x, y ) );
                double ir_denominator = 2*m.transfer_impedance( rect_x, rect_y, x, y );
                if ( ir_denominator == 0 ) {
                    continue;
                }

                // make denominator positive
                if ( ir_denominator < 0 ) {
                    ir_denominator = -ir_denominator;
                    ir_numerator = -ir_numerator;
                }

                // calculate the voltage between rect_x and rect_y
                // vr = transfer_impedance( x, y, rect_x, rect_y ) + ir*transfer_impedance( rect_x, rect_y, rect_x, rect_y )
                double vr = ( m.transfer_impedance( x, y, rect_x, rect_y )
                          + ir_numerator / ir_denominator * m.transfer_impedance( rect_x, rect_y, rect_x, rect_y ) ) / determinant;

                // i_combined is combined current of rect resistor and current source
                double i_combined = vr - ir_numerator/ir_denominator;
                if ( i_combined == 0 || vr == 0 )
                    continue;
                double aspect_ratio = fabs(vr) > fabs(i_combined) ? vr/i_combined : i_combined/vr;
                if ( aspect_ratio < 0 )
                    // check failed.
                    continue;
                
                // check the aspect ratio against the filter list.  Order of current APS is num_edges-2
                if ( max_square_order && !filter.check_aspect_ratio( aspect_ratio,  max_square_order - (graph.num_edges-2) ) )
                    continue;

                // ###############################################
                // end of critical performance section
                // ###############################################

                double t = gcd( ir_denominator, ir_numerator );
                ir_denominator /= t;
                ir_numerator /= t;

                // now calculate node voltages.
                // all voltages are actually multiplied by determinant*ir_denominator to make them integer.
                double voltage[64];
                for ( u32 j =0; j<num_nodes; ++j )  {
                    voltage[j] =   ir_denominator * ( m.select(j,x) - m.select(j,y) )
                                 + ir_numerator * ( m.select(j, rect_x) - m.select(j, rect_y ) );
                }

                // find greatest common divisor of the voltages
                double g = 0;
                for ( u32 j =0; j<num_nodes; ++j )  {
                    g = gcd( voltage[j], g );
                }

                // divide through by g
                for ( u32 j =0; j<num_nodes; ++j )  {
                    voltage[j] = voltage[j]/g;
                    assert( voltage[j] == round( voltage[j] ) );
                }

                // compute square sizes
                struct Square_t {
                    double width;
                    double height;
                    double xpos;
                    double ypos;
                } square[128];
                u32 num_squares = 0;
                for ( Edge* e = graph.edge_list; e; e = e->next ) {
                    if ( e == terminal_edge || e == rectangle_edge )
                        continue;
                    u32 n0 = e->dedge_0.node_num;
                    u32 n1 = e->dedge_1.node_num;
                    double size = fabs( voltage[ n0 ] - voltage[ n1 ] );
                    square[num_squares].width = size;
                    square[num_squares++].height = size;
                }

                // check if all different
                bool perfect = true;
                for ( u32 i = 0; i< num_squares; ++i ) {
                    if ( square[i].width == 0. ) {
                        perfect = false;
                        break;
                    }
                    for ( u32 j = 0; j< i; ++j ) {
                        if (  square[i].width == square[j].width )  {
                            perfect = false;
                            break;
                        }
                    }
                    if ( !perfect ) {
                        break;
                    }
                }

                if ( !perfect ) {
                    continue;
                }

                // at this point we have an APS.  Print it.
                //
                // compute "potential" for each face.  Each node of graph
                // represents horizontal line in PR and its voltage represents
                // its vertical position.  Each face of graph represents a vertical
                // line in the square and its potential represents its horizontal
                // position.

                double face_potential[64];
                bool done=false;
                u64 face_potential_computed = 0;
                u32 left_face = terminal_edge->dedge_0.face_num;
                face_potential[left_face] = 0.;
                face_potential_computed |= 1uLL << left_face;

                while( !done ) {
                    for ( Edge* ee = graph.edge_list; ee; ee=ee->next ) {
                        if ( ee == terminal_edge || ee == rectangle_edge )
                            continue;
                        u32 face0 = ee->dedge_0.face_num;
                        u32 face1 = ee->dedge_1.face_num;
                        u32 node0 = ee->dedge_0.node_num;
                        u32 node1 = ee->dedge_1.node_num;
                        if (   !( face_potential_computed & 1uLL << face0 )
                             && ( face_potential_computed & 1uLL << face1 ) ) {
                            // swap faces and nodes
                            u32 temp = face0;
                            face0 = face1;
                            face1 = temp;
                            temp = node0;
                            node0 = node1;
                            node1 = temp;
                        }
                        if (    ( face_potential_computed & 1uLL << face0 )
                             &&!( face_potential_computed & 1uLL << face1 ) ) {
                            face_potential[ face1 ] = face_potential[ face0 ]
                                                    + voltage[ node1 ]
                                                    - voltage[ node0 ];
                            face_potential_computed |= 1uLL << face1;
                            if ( face_potential_computed + 1 == 1uLL << graph.num_faces ) {
                                done = true;
                                break;
                            }
                        }
                    }
                }

                // Add in the one rectangle.
                u32 rn0 = rectangle_edge->dedge_0.node_num;
                u32 rn1 = rectangle_edge->dedge_1.node_num;
                u32 rf0 = rectangle_edge->dedge_0.face_num;
                u32 rf1 = rectangle_edge->dedge_1.face_num;
                square[num_squares].height = fabs( voltage[ rn0 ] - voltage[ rn1 ] );
                square[num_squares].width = fabs( face_potential[ rf0 ] - face_potential[ rf1 ] );
                ++num_squares;
                
                // at this point we can compute the X/Y position of each component square
                // (and the rectangle) within the APS.
                // THe squares are in linked list order except rectangle which is last
                u32 top_node = terminal_edge->dedge_0.node_num;
                u32 bottom_node = terminal_edge->dedge_1.node_num;
                double vbottom = voltage[ bottom_node ];
                {
                    u32 i = 0;
                    for ( Edge* ee = graph.edge_list; ee; ee=ee->next ) {
                        if ( ee == terminal_edge || ee == rectangle_edge )
                            continue;
                        u32 node0 = ee->dedge_0.node_num;
                        u32 node1 = ee->dedge_1.node_num;
                        u32 face0 = ee->dedge_0.face_num;
                        u32 face1 = ee->dedge_1.face_num;
                        square[i].ypos = fmin( voltage[node0], voltage[node1] ) - vbottom;
                        square[i].xpos = fmin( face_potential[face0], face_potential[face1] );
                        ++i;
                    }

                    u32 rnode0 = rectangle_edge->dedge_0.node_num;
                    u32 rnode1 = rectangle_edge->dedge_1.node_num;
                    u32 rface0 = rectangle_edge->dedge_0.face_num;
                    u32 rface1 = rectangle_edge->dedge_1.face_num;
                    square[i].ypos = fmin( voltage[rnode0], voltage[rnode1] ) - vbottom;
                    square[i].xpos = fmin( face_potential[rface0], face_potential[rface1] );
                }
                double total_height = voltage[ top_node ] - vbottom;
                double total_width = determinant*ir_denominator/g - total_height;
                assert( total_height == total_width );

                // reject duals
                if ( graph.num_nodes == graph.num_faces ) {
                    // number of nodes and faces is swapped when
                    // creating a dual, so we always reject graphs
                    // with more nodes than faces.
                    // But when num_nodes and num_faces are equal, we
                    // need a tie breaker.

                    // find the size of the four corner squares.
                    // use zero for size of rectangle.
                    double A = -1.;
                    double B = -1.;
                    double C = -1.;
                    double D = -1.;
                    for ( u32 i = 0; i < num_squares; ++i ) {
                        double x = square[i].xpos;
                        double y = square[i].ypos;
                        double width = square[i].width;
                        double height = square[i].height;
                        if ( x == 0. && y == 0. )
                            A = width==height ? width : 0;
                        else if ( x == 0. && y+height == total_height )
                            B = width==height ? width : 0;
                        else if ( x+width == total_width && y == 0. )
                            C = width==height ? width : 0;
                        else if ( x+width == total_width && y+height == total_height )
                            D = width==height ? width : 0;
                    }
                    assert( A >= 0. );
                    assert( B >= 0. );
                    assert( C >= 0. );
                    assert( D >= 0. );

                    // the order of the sizes of the corners will be either an even or
                    // an odd permutation of the sorted order.  Of the rectangle and
                    // its dual, one will be even and one odd, so reject the odd one.
                    if ( (A>B) ^ (A>C) ^ (A>D) ^ (B>C) ^ (B>D) ^ (C>D) )
                        continue;
                }

                // now sort the squares
                for ( u32 i = 1; i < num_squares; ++i ) {
                    for ( u32 j = i; j >= 1; --j ) {
                        if ( square[j].ypos < square[j-1].ypos
                                || (    square[j].ypos == square[j-1].ypos
                                    &&  square[j].xpos  < square[j-1].xpos  ) ) {
                            // swap
                            Square_t temp = square[j];
                            square[j] = square[j-1];
                            square[j-1] = temp;
                        } else {
                            break;
                        }
                    }
                }
                // print the bouwkamp code.
                printf("%d %g %g (", num_squares-1, total_width, total_height );
                for ( u32 i = 0; i < num_squares; ++i ) {
                    double width = square[i].width;
                    double height = square[i].height;
                    if ( width == height ) 
                         printf("%g", width );
                    else
                         printf("%g:%g", width, height );
                     // now print separator
                     if ( i == num_squares-1 ) {
                         printf(")");
                     } else if ( square[i].ypos < square[i+1].ypos ) {
                         printf(")(");
                     } else {
                         printf(",");
                     }
                }
                printf("\n");
            }
        }
    }
}

