#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <ctype.h>
#include <unistd.h>

#include "Boundary.h"


// each component square is defined by its size and x/y position
// If we traverse the perimeter of an embedded rectangle in a counter clockwise direction,
// each component square will be  followed by another.  The next pointers point to the next
// component square on the perimeter of the rectangle, either on the bottom edge of
// the rectangle, or left, right, or top edge.  
struct ComponentSquare {
    int size;
    int x;
    int y;

    ComponentSquare* next_on_bottom;
    ComponentSquare* next_on_right;
    ComponentSquare* next_on_top;
    ComponentSquare* next_on_left;
};

// A squared rectangle contains an order, height, width, and list of components.
struct SquaredRectangle {
    int order;
    int width;
    int height;
    ComponentSquare cs[100];
};

// this describes a rectangle embedded within a squared square.
struct Rectangle {
    int left;
    int right;
    int bottom;
    int top;
};


// this is a quick routine to check validity of a SquaredRectangle
// it checks:
//      all components are within overall boundary
//      no two components overlap
//      sum of areas of components equals overall square size
// Used for debugging
void check_rectangle( SquaredRectangle& s )
{
    int area = 0;
    int order = s.order;
    for ( int i=0; i < order; ++i ) {
        int size = s.cs[i].size;
        int x = s.cs[i].x;
        int y = s.cs[i].y;
        area += size*size;
        assert( x >= 0 && y >=0 && x+size <= s.width && y+size <= s.height );
        // now check all other squares for overlap and unequal size
        for ( int j = i+1; j < order; ++ j ) {
            int size2 = s.cs[j].size;
            int x2 = s.cs[j].x;
            int y2 = s.cs[j].y;
            assert( x+size <= x2  ||  x2+size2 <= x  ||  y+size <= y2  || y2+size2 <= y );
            if ( size == size2 ) {
                fprintf( stderr, "Error - Not Perfect.\n" );
                assert(0);
            }
        }
    }
    assert( area == s.width * s.height );
}

// reads an input string of either Bouwkamp or table code.
void input_sr( char* string, SquaredRectangle& sr )
{
    int n;
    char *p = string;
    int status;
    Boundary boundary;

    status = sscanf( p, "%d%n", &sr.order, &n );
    assert( status == 1 );
    p += n;
    status = sscanf( p, "%d %d%n", &sr.width, &sr.height, &n );
    assert( status == 2 );
    p += n;
    boundary.initialize( sr.width, sr.height );
    for ( int i = 0; i < sr.order; ++i ) {
        while( *p && !isdigit(*p) ) 
            ++p;
        status = sscanf( p, "%d%n", &sr.cs[i].size, &n );
        assert( status == 1 );
        p += n;

        // Boundary class keep track of geomerty.
        // indicates X/Y coordinates of next component square to be added.
        boundary.get_loc_lowest( sr.cs[i].x, sr.cs[i].y );
        boundary.add_square(  sr.cs[i].x,
                              sr.cs[i].x + sr.cs[i].size,
                              sr.cs[i].y,
                              sr.cs[i].y + sr.cs[i].size );
    }
    check_rectangle( sr );
}

    
// check if a given component square is within a defined rectangle
// returns 1 if inside rectangle, 0 if outside rectangle,
int check_component_in_rectangle( ComponentSquare &cs, Rectangle &r )
{
    int size = cs.size;
    int x = cs.x;
    int y = cs.y;
    if ( x >= r.left && x+size <= r.right && y >= r.bottom && y+size <= r.top )
        return 1;
    else if ( x+size <= r.left || x >= r.right || y+size <= r.bottom || y >= r.top )
        return 0;
    else
        // this would be an error.
        assert(0);
}

// updates the pointers to allow traversing the perimeter of an embedded rectangle.
// These pointers are independent of what rectangle we are looking at so they can
// be set up once at the beginning.
void update_next_in_rectangle_pointers( SquaredRectangle& s ) 
{
    int order = s.order;
    for ( int i = 0; i < order; ++i ) {
        ComponentSquare &cs = s.cs[i];
        cs.next_on_bottom = 0;
        cs.next_on_right = 0;
        cs.next_on_top = 0;
        cs.next_on_left = 0;
        for ( int j = 0; j < order; ++j ) {
            ComponentSquare &cs2 = s.cs[j];
            if ( cs2.y > cs.y + cs.size ) {
                // we are done with cs.  Move on to next i.
                break;
            }
            if ( cs.y == cs2.y && cs.x + cs.size == cs2.x )
                cs.next_on_bottom = &cs2;
            if ( cs.x + cs.size == cs2.x + cs2.size && cs.y + cs.size == cs2.y )
                cs.next_on_right = &cs2;
            if ( cs.y + cs.size == cs2.y + cs2.size && cs.x == cs2.x + cs2.size )
                cs.next_on_top = &cs2;
            if ( cs.x == cs2.x && cs.y == cs2.y + cs2.size )
                cs.next_on_left = &cs2;
        }
    }
}


void swap_cs( ComponentSquare &x, ComponentSquare &y ) {
    ComponentSquare temp = x;
    x = y;
    y = temp;
}

// sort the components in a squared square
// primary sort on Y and secondary sort on X
// This function is in  the  performance path,
// however because of the moderately small size of the array, and
// because it is usually partially sorted, there is nothing
// better than an insertion sort here.
void sort_squared_rectangle( SquaredRectangle& sr )
{
    int order = sr.order;
    for ( int i = 1; i < order; ++i ) {
        for ( int j = i; j>=1; j -= 1) {
            if (     sr.cs[j].y < sr.cs[j-1].y
                || ( sr.cs[j].y == sr.cs[j-1].y && sr.cs[j].x < sr.cs[j-1].x ) ) {

                swap_cs( sr.cs[j], sr.cs[j-1] );
            } else {
                break;
            }
        }
    }
    // uncomment this for debug.  Too slow to leave here.
    // check_rectangle( sr );
}


// compares two isomers.  expects them to be sorted.
// Returns -1 if first is less than second, 0 if equal,
// 1 if first greater than second.
// preferred isomer is largest.
int compare_isomers( SquaredRectangle& sr1, SquaredRectangle& sr2 )
{
    int order = sr1.order;
    assert( order == sr2.order );
    for ( int i = 0; i < order; ++i ) {
        if ( sr1.cs[i].size < sr2.cs[i].size )
            return -1;
        else if ( sr1.cs[i].size > sr2.cs[i].size )
            return 1;
    }
    return 0;
}


void reflect_horizontal(  SquaredRectangle& sr, Rectangle& r ) 
{
    int order = sr.order;
    for ( int i = 0; i < order; ++i ) {
        int cir = check_component_in_rectangle( sr.cs[i], r );
        if ( cir ) {
            sr.cs[i].x = r.left + ( r.right - sr.cs[i].x - sr.cs[i].size );
        }
    }
    sort_squared_rectangle( sr );
}


void reflect_vertical(  SquaredRectangle& sr, Rectangle &r ) 
{
    int order = sr.order;
    for ( int i = 0; i < order; ++i ) {
        int cir = check_component_in_rectangle( sr.cs[i], r );
        if ( cir ) {
            sr.cs[i].y = r.bottom + ( r.top - sr.cs[i].y - sr.cs[i].size );
        }
    }
    sort_squared_rectangle( sr );
}


void reflect_diagonal(  SquaredRectangle& sr, Rectangle &r ) 
{
    // diagonal reflection can be done only on squares
    assert( r.right-r.left == r.top-r.bottom );
    int order = sr.order;
    for ( int i = 0; i < order; ++i ) {
        int cir = check_component_in_rectangle( sr.cs[i], r );
        if ( cir ) {
            int x = sr.cs[i].x;
            int y = sr.cs[i].y;
            sr.cs[i].x = r.left + ( y - r.bottom );
            sr.cs[i].y = r.bottom + ( x - r.left );
        }
    }
    sort_squared_rectangle( sr );
}

// figure out if SquaredRectangle "sr" is in preferred orientation.
bool preferred_orientation(  SquaredRectangle& sr )
{
    int width = sr.width;
    int height = sr.height;
    int order = sr.order;
    int corner_size = sr.cs[0].size;
    int cha_size; // size of horizontally adjacent to first corner

    cha_size = sr.cs[1].size;
    // do some quick sanity checks.
    assert( sr.cs[0].x == 0 && sr.cs[0].y == 0 && sr.cs[1].x == sr.cs[0].size && sr.cs[1].y == 0 );
    for ( int i = 1; i < order; ++i ) {
        int cx = sr.cs[i].x;
        int cy = sr.cs[i].y;
        int csize = sr.cs[i].size;
        if ( cx == 0 && cy+csize == height && csize > corner_size ) {
            // upper left corner bigger
            return false;
        }
        if ( cx+csize == width && cy == 0 && csize > corner_size ) {
            // lower right corner bigger
            return false;
        }
        if ( cx+csize == width && cy+csize == height && csize > corner_size ) {
            // upper right corner bigger
            return false;
        }
        if ( cx == 0 && cy == corner_size && csize > cha_size && width == height ) {
            // vertically adjacent square to first corner is larger than horizontally adjacent square
            // applies only to squared squares
            return false;
        }
    }
    return true;
}


#define MAX_ISOMERS 1000000

int main(int argc, char* argv[] )
{
    // create array to store isomers.
    SquaredRectangle *isomers = (SquaredRectangle*)malloc( MAX_ISOMERS*sizeof( SquaredRectangle ) );
    // create index array. this maintains a sorted index into isomers.
    SquaredRectangle **isomer_index = (SquaredRectangle**)malloc( MAX_ISOMERS*sizeof( SquaredRectangle* ) );

    // figure out the command line options
    int opt;
    bool bouwkamp = false;
    bool all_isomers = false;
    bool print_isomer_count = false;
    bool include_rotations_and_reflections = false;
    bool simple_only = false;
    bool compound_only = false;
    while ( (opt = getopt( argc, argv, "abcrSC" ) ) != -1 ) {
        if ( opt == 'a' )
            all_isomers = true;
        else if ( opt == 'b' )
            bouwkamp = true;
        else if ( opt == 'c' )
            print_isomer_count = true;
        else if ( opt == 'r' ) 
            include_rotations_and_reflections = true;
        else if ( opt == 'S' ) 
            simple_only = true;
        else if ( opt == 'C' ) 
            compound_only = true;
    }

    // loop through input lines
    char* line = 0;
    int status;
    size_t len;
    while ( ( status = getline( &line, &len, stdin ) ) != -1 ) {
        // read the line into the first isomer
        input_sr( line, isomers[0] );
        int num_isomers = 1;
        isomer_index[0] = &isomers[0];

        // loop through all the isomers.
        // intially there is only one, but that will increase as we find more
        for ( int current_isomer = 0; current_isomer < num_isomers; ++ current_isomer ) {
            // look for embedded rectangles
            SquaredRectangle &sr = isomers[current_isomer];
            int order = sr.order;
            update_next_in_rectangle_pointers( sr );
            bool current_preferred_orientation = preferred_orientation( sr );
            for ( int i = 0; i < order; ++i ) {
                if ( !current_preferred_orientation && i > 0 ) 
                    // save time - only look at reorientation of the entier square for non-preferred orientations.
                    break;
                // try each component square successively as the lower left corner of the rectange
                // and try to walk around the perimeter
                ComponentSquare *lower_left = &sr.cs[i];
                for ( ComponentSquare *lower_right = lower_left; lower_right; lower_right = lower_right->next_on_bottom ) {
                    for ( ComponentSquare *upper_right = lower_right; upper_right; upper_right = upper_right->next_on_right ) {
                        if ( upper_right == lower_left )
                            // skip embedded rectangles that consist of a single component square
                            continue;
                        if (      !current_preferred_orientation 
                             && ( upper_right->x + upper_right->size != sr.width  || upper_right->y + upper_right->size != sr.height ) )  {
                            // if non-preferred orientation, skip this rectangle unless upper right component is upper
                            // right of entire square.
                            continue;
                        }

                        ComponentSquare *upper_left = upper_right;
                        while ( upper_left && upper_left->x > lower_left->x )
                            upper_left = upper_left->next_on_top;
                        if ( upper_left == 0 || upper_left->x != lower_left->x )
                            // failed to find upper left corner
                            continue;

                        ComponentSquare *p = upper_left;
                        while ( p && p!=lower_left)
                            p = p->next_on_left;
                        if ( p!= lower_left )
                            // faild on left side
                            continue;

                        // found an embedded rectngle
                        Rectangle r;
                        r.left = lower_left->x;
                        r.right = upper_right->x + upper_right->size;
                        r.bottom = lower_left->y;
                        r.top = upper_right->y + upper_right->size;


                        // at this point we have a valid rectangle (possibly the entire rectangle)
                        // try each of the three reflections to see if we can create new isomer.
                        for ( int reflection_type = 0; reflection_type < 3; ++reflection_type ) {
                            isomers[num_isomers] = isomers[current_isomer];
                            switch (reflection_type ) {
                            case 0: 
                                reflect_horizontal( isomers[num_isomers], r );
                                break;
                            case 1: 
                                reflect_vertical( isomers[num_isomers], r );
                                break;
                            case 2: 
                                if ( r.right-r.left == r.top-r.bottom ) 
                                    // diagonal reflection done only on squares.
                                    reflect_diagonal( isomers[num_isomers], r );
                                break;
                            }

                            // Now compare the new isomer to all the ones on the list
                            // and increment num_isomers only if it is not a duplicate.
                            // Use binary search to find first either isomer equal to new one
                            // or insertion point.
                            bool duplicate = false;
                            int begin = 0;
                            int end = num_isomers - 1;
                            while ( end >= begin ) {
                                int a = ( begin + end )/2;
                                int comp_result = compare_isomers( *isomer_index[a], isomers[num_isomers] );
                                if ( comp_result == 0 ) {
                                    duplicate = true;
                                    break;
                                }
                                if ( comp_result > 0 ) {
                                    begin = a+1;
                                } else {
                                    end = a-1;
                                }
                            }
                            if ( !duplicate ) {
                                // now we need to insert a pointer to the new isomer in the index array
                                // "begin" should now point to the location where this pointer belongs.
                                for ( int j = num_isomers; j > begin; --j ) {
                                    isomer_index[j] = isomer_index[j-1];
                                }
                                isomer_index[begin] = &isomers[num_isomers];
                                ++ num_isomers;
                                if ( num_isomers >= MAX_ISOMERS ) {
                                    fprintf( stderr, "Exceeded MAX_ISOMERS %d\n", MAX_ISOMERS );
                                    assert( 0 );
                                }
                            }
                        }
                    }
                }
            }
        }
        // at this point we have the complete isomer list.
        // print isomers
        int instance = 1;
        int isomer_count = 0;
        for ( int j = 0; j < num_isomers; ++j ) {
            if ( preferred_orientation ( isomers[j] ) ) {
                ++ isomer_count;
            }
        }
        assert( num_isomers == isomer_count * ( isomers[0].width == isomers[0].height ? 8 : 4 ) );
        if ( isomer_count == 1 ? !compound_only : !simple_only ) {
            for ( int i = 0; i < num_isomers; ++i ) {
                if ( !include_rotations_and_reflections && !preferred_orientation( *isomer_index[i] ) )
                    continue;
                printf( "%d %d %d ", isomer_index[i]->order, isomer_index[i]->width, isomer_index[i]->height );
                if ( bouwkamp )
                    printf( "(" );
                for ( int j = 0; j < isomer_index[i]->order; ++j ) {
                    printf( "%d", isomer_index[i]->cs[j].size );
                    if ( bouwkamp ) {
                        if ( j == isomer_index[i]->order-1 )
                            printf( ")" );
                        else {
                            if ( isomer_index[i]->cs[j].y == isomer_index[i]->cs[j+1].y )
                                printf( "," );
                            else 
                                printf( ")(" );
                        }
                    } else {
                        printf( " " );
                    }
                }
                if ( print_isomer_count ) {
                    assert( num_isomers == isomer_count * ( isomers[0].width == isomers[0].height ? 8 : 4 ) );
                    printf( " * (%d/%d)", instance, isomer_count );
                    ++instance;
                }
                printf("\n");

                if ( !all_isomers )
                    break;
            }
        }
    }
}
