#ifndef GRAPH_H
#define GRAPH_H

#include "Types.h"

enum ReductionType {
    OPEN,
    SHORT,
    DIAMOND };

struct OEdge {
    struct DEdge *parent;
    struct Edge  *grandparent;

    OEdge *next;
    OEdge *next_on_node;
    OEdge *next_on_face;
    OEdge *other_end;
    u32* face_num_p;

    u32 hash[2];

    // compare operators are critical functionality.
    // Objects should compare equal only if there is 
    // full isomorphic mapping.  Otherwise comparison
    // is arbitrary but should be transitive.
    //
    // Make comparison monotonically increasing with node
    // size and monotonically decreasing with face size.
    // This allows some efficiencies in screening possible
    // edge additions.
    int compare( OEdge &x, bool first );
    bool operator==( OEdge &x ) {return compare(x, true) == 0;};
    bool operator!=( OEdge &x ) {return compare(x, true) != 0;};
    bool operator> ( OEdge &x ) {return compare(x, true) > 0;};
    bool operator< ( OEdge &x ) {return compare(x, true) < 0;};
    bool operator>=( OEdge &x ) {return compare(x, true) >= 0;};
    bool operator<=( OEdge &x ) {return compare(x, true) <= 0;};

    void print_graph( char* output_string );
};

struct DEdge {
    DEdge *cl_next;
    DEdge *cc_next;
    DEdge *other_end;
    DEdge *next;
    struct Edge *parent;
    u32 node_num;
    u32 face_num;

    OEdge unreflected;
    OEdge reflected;

    void update_oedge_pointers();
    bool node_connects_to( u32 node_num );

};

struct Edge {
    static Edge *free_list;
    Edge *next;
    struct Graph *graph;

    // allocate or free a pair.
    static Edge *allocate();
    void free();
    
    void update_next( Edge* new_next );

    DEdge dedge_0;
    DEdge dedge_1;
};

const u32 MAX_GRAPH_SIZE=100;

struct Graph {

    u32 num_edges;
    u32 num_nodes;
    u32 num_faces;

    u32 node_degree[ MAX_GRAPH_SIZE ];
    u32 face_degree[ MAX_GRAPH_SIZE ];

    DEdge *node_edge[ MAX_GRAPH_SIZE ];
    DEdge *face_edge[ MAX_GRAPH_SIZE ];

    Edge *edge_list;
    DEdge *dedge_list;
    OEdge *oedge_list;

    Graph();
    void connect_edge_cl( DEdge* new_edge, DEdge *old_edge );
    void connect_edge_cc( DEdge* new_edge, DEdge *old_edge );
    void disconnect_edge( DEdge* e );
    void insert_edge( DEdge* new_edge, DEdge* neighbor, u32 other_node );
    void remove_open( Edge* e );
    void remove_short( Edge* e );
    void remove_diamond( Edge* e );
    void add_open( DEdge* ref0, DEdge* ref1 );
    void add_short( DEdge* ref0, DEdge* ref1 );
    void add_diamond( Edge* e );
    void initialize( );
    void check_valid();
    void create_from_string( const char* s );
    void debug_print();
    bool check_3_connected();
    bool check_2_connected();
    void check_simple_mindeg_3();
    bool check_open_3con( Edge *e );
    bool check_short_3con( Edge *e );
    bool check_open_2con( Edge *e );
    bool check_short_2con( Edge *e );
    bool check_diamond_ok( Edge *e );
    bool is_wheel();
    void remove_edge_from_linked_lists( Edge *e );
    bool check_preferred_reduction( OEdge* e, ReductionType reduction_type, u32 connectivity );
    void create_wheel( u32 size );
    OEdge* max_oedge( DEdge* d );
    OEdge* max_oedge( Edge* e );
    OEdge* max_oedge();
    u32 compute_hash();
};



struct StringSet {
    u32 size;
    char* set[256];

    StringSet();
    ~StringSet() { reset(); };

    bool check_and_include( char* string );
    void reset();
    char* allocate_string();
    void free_string( char* string );
    static char* free_list;
};




#endif // GRAPH_H
