#include <stdio.h>
#include <assert.h>
#include <string.h>
#include <stdlib.h>
#include "Graph.h"

Edge* Edge::free_list;

Edge* Edge::allocate()
{
    Edge *ret_val;
    if ( free_list ) {
        ret_val = free_list;
        free_list = free_list->next;
        ret_val->next = 0;
    } else {
        ret_val = new Edge;
        assert( ret_val );

        // initialize fixed fields
        // These fields never change even when allocating deallocating the edge
        // so we get them for free, computationally.
        ret_val->dedge_0.other_end = &ret_val->dedge_1;
        ret_val->dedge_0.next = &ret_val->dedge_1;
        ret_val->dedge_0.parent = ret_val;

        ret_val->dedge_1.other_end = &ret_val->dedge_0;
        ret_val->dedge_1.parent = ret_val;

        ret_val->dedge_0.unreflected.next = &ret_val->dedge_0.reflected;
        ret_val->dedge_0.unreflected.parent = &ret_val->dedge_0;
        ret_val->dedge_0.unreflected.grandparent = ret_val;
        ret_val->dedge_0.unreflected.other_end = &ret_val->dedge_1.unreflected;

        ret_val->dedge_0.reflected.next = &ret_val->dedge_1.unreflected;
        ret_val->dedge_0.reflected.parent = &ret_val->dedge_0;
        ret_val->dedge_0.reflected.grandparent = ret_val;
        ret_val->dedge_0.reflected.other_end = &ret_val->dedge_1.reflected;

        ret_val->dedge_1.unreflected.next = &ret_val->dedge_1.reflected;
        ret_val->dedge_1.unreflected.parent = &ret_val->dedge_1;
        ret_val->dedge_1.unreflected.grandparent = ret_val;
        ret_val->dedge_1.unreflected.other_end = &ret_val->dedge_0.unreflected;

        ret_val->dedge_1.reflected.parent = &ret_val->dedge_1;
        ret_val->dedge_1.reflected.grandparent = ret_val;
        ret_val->dedge_1.reflected.other_end = &ret_val->dedge_0.reflected;

        ret_val->dedge_0.unreflected.face_num_p = &ret_val->dedge_0.face_num;
        ret_val->dedge_0.reflected.face_num_p = &ret_val->dedge_1.face_num;
        ret_val->dedge_1.unreflected.face_num_p = &ret_val->dedge_1.face_num;
        ret_val->dedge_1.reflected.face_num_p = &ret_val->dedge_0.face_num;
    }
    return ret_val;
}


void Edge::free() 
{
    next = free_list;
    free_list = this;
}

// this routine should be called whenever next is modified.
void Edge::update_next( Edge* new_next )
{
    // update pointers
    next = new_next;
    dedge_1.next = next ? &next->dedge_0 : 0;
    dedge_1.reflected.next = next ? &next->dedge_0.unreflected : 0;
}

Graph::Graph() { edge_list= 0; }

void Graph::connect_edge_cl( DEdge* new_edge, DEdge *old_edge ) 
{
    old_edge->cl_next->cc_next = new_edge;
    new_edge->cl_next = old_edge->cl_next;
    new_edge->cc_next = old_edge;
    old_edge->cl_next = new_edge;
    new_edge->update_oedge_pointers();
}

void Graph::connect_edge_cc( DEdge* new_edge, DEdge *old_edge ) 
{
    old_edge->cc_next->cl_next = new_edge;
    new_edge->cc_next = old_edge->cc_next;
    new_edge->cl_next = old_edge;
    old_edge->cc_next = new_edge;
    new_edge->update_oedge_pointers();
}

void Graph::disconnect_edge( DEdge* e )
{
    // move node_edge if needed
    u32 node_num = e->node_num;
    DEdge*& current_node_edge =  node_edge[node_num];
    if ( current_node_edge == e ) {
        current_node_edge = e->cl_next;
    }

    e->cc_next->cl_next = e->cl_next;
    e->cl_next->cc_next = e->cc_next;

    // udpate oedge pointers
    e->cl_next->other_end->unreflected.next_on_face = &e->cc_next->unreflected;
    e->cc_next->other_end->  reflected.next_on_face = &e->cl_next->  reflected;
    e->cl_next->  reflected.next_on_node = &e->cc_next->  reflected;
    e->cc_next->unreflected.next_on_node = &e->cl_next->unreflected;

    e->cl_next = 0;
    e->cc_next = 0;
}

void DEdge::update_oedge_pointers()
{
    unreflected.next_on_node = &cl_next->unreflected;
    reflected.next_on_node = &cc_next->reflected;
    other_end->unreflected.next_on_face = &cc_next->unreflected;
    other_end->reflected.next_on_face = &cl_next->reflected;

    cc_next->unreflected.next_on_node = &unreflected;
    cl_next->reflected.next_on_node = &reflected;
    cl_next->other_end->unreflected.next_on_face = &unreflected;
    cc_next->other_end->reflected.next_on_face = &reflected;
}

bool DEdge::node_connects_to( u32 node_num ) 
{
    DEdge* y = this;
    do {
        if ( y->other_end->node_num == node_num )
            return true;
        y = y->cl_next;
    } while ( y != this );
    return false;
}

// insert an edge into a graph.
// This function used only when creating a graph
// from a string.
// New edge is added clockwise from neighbor
void Graph::insert_edge( DEdge* new_edge, DEdge* neighbor, u32 other_node )
{
    connect_edge_cl( new_edge, neighbor );
    new_edge->node_num = neighbor->node_num;
    new_edge->face_num = neighbor->face_num;
    new_edge->other_end->node_num = other_node;
    ++node_degree[ neighbor->node_num ];
    if ( other_node < num_nodes ) {
        // connect to existing node
        DEdge *p;
        int count = 0;
        for ( p = neighbor->other_end; p->node_num != other_node; p = p->cc_next->other_end ) {
            ++count;
            assert( count < 1000 );
        }
        connect_edge_cc( new_edge->other_end, p );
        ++node_degree[ other_node ];

        // create a new face
        p = neighbor;
        face_degree[ num_faces ] = 0;
        do {
            p->face_num = num_faces;
            p = p->other_end->cc_next;
            ++face_degree[ num_faces ];
        } while ( p != neighbor );
        face_degree[ new_edge->face_num ] -= face_degree[ num_faces ] - 2;
        face_edge[ num_faces ] = neighbor;
        face_edge[ new_edge->face_num ] = new_edge;
        ++ num_faces;
    } else {
        // connecting to new node
        assert( other_node == num_nodes );
        DEdge *p = new_edge->other_end;
        p->node_num = num_nodes;
        node_degree[ num_nodes ] = 1;
        face_degree[ neighbor->face_num ] += 2;
        node_edge[ num_nodes ] = p;
        ++num_nodes;
        p->face_num = neighbor->face_num;
        p->cc_next = p;
        p->cl_next = p;
        p->update_oedge_pointers();
    }

    // add to overall edge list
    new_edge->parent->update_next( edge_list );
    edge_list = new_edge->parent;
    dedge_list = &edge_list->dedge_0;
    oedge_list = &edge_list->dedge_0.unreflected;
    new_edge->parent->graph = this;
    ++num_edges;
}


// initializes with a single edge from node 0 to node 1
void Graph::initialize( )
{
    // cleanup any old contents;
    while ( edge_list ) {
        Edge *temp = edge_list;
        edge_list = temp->next;
        temp->free();
    }
    num_edges = 1;
    num_nodes = 2;
    num_faces = 1;

    for ( u32 i = 0; i < MAX_GRAPH_SIZE; ++i ) {
        node_degree[i] = 0;
        node_edge[i] = 0;
        face_degree[i] = 0;
        face_edge[i] = 0;
    }

    edge_list = Edge::allocate();
    dedge_list = &edge_list->dedge_0;
    oedge_list = &edge_list->dedge_0.unreflected;
    edge_list->graph = this;

    DEdge *p = &edge_list->dedge_0;
    p->node_num = 0;
    p->face_num = 0;
    p->cl_next = p;
    p->cc_next = p;
    p = &edge_list->dedge_1;
    p->node_num = 1;
    p->face_num = 0;
    p->cl_next = p;
    p->cc_next = p;

    node_degree[0] = 1;
    node_degree[1] = 1;
    node_edge[0] = &edge_list->dedge_0;
    node_edge[1] = &edge_list->dedge_1;
    face_degree[0] = 2;
    face_edge[0] = &edge_list->dedge_0;

    edge_list->update_next( 0 );
    edge_list->dedge_0.update_oedge_pointers();
    edge_list->dedge_1.update_oedge_pointers();
}


// for debugging - check that graph is fully valid
// very slow.
void Graph::check_valid()
{
    // first check edge list
    Edge *e = edge_list;
    for ( u32 i = 0; i < num_edges; ++i ) {
        assert( i < 1000 );
        assert( e );
        assert( e->dedge_0.parent == e );
        assert( e->dedge_1.parent == e );
        e = e->next;
    }
    assert( e == 0 );
        
    // check DEdge list
    DEdge *p = dedge_list;
    for ( u32 i = 0; i < num_edges*2; ++i ) {
        assert( i < 1000 );
        assert( p );
        assert( p->other_end );
        assert( p->other_end->other_end == p );
        assert( p->unreflected.parent == p );
        assert( p->reflected.parent == p );
        assert( p->unreflected.grandparent == p->parent );
        assert( p->reflected.grandparent == p->parent );
        p = p->next;
    }
    assert( p == 0 );

    
    // check OEdge list
    OEdge *r = oedge_list;
    for ( u32 i = 0; i < num_edges*4; ++i ) {
        assert( i < 1000 );
        assert( r->other_end->other_end == r );
        r = r->next;
    }
    assert( r == 0 );

    // global checks
    assert( num_faces + num_nodes == num_edges + 2 );
    u64 node_check = 0;
    u64 face_check = 0;

    // Iterate through all edges and check correct.
    for ( p = dedge_list; p; p = p->next ) {
        for ( int i = 0; i < 2 ; ++i ) {
            //check node
            u32 n = p->node_num;
            node_check |= 1uLL << n;
            DEdge *q=p;
            assert( node_degree[p->node_num] > 0 );
            for (u32 i = 0; i < node_degree[p->node_num]; ++i ) {
                assert( q->node_num == p->node_num );
                assert( q->cc_next->cl_next == q );
                assert( i==0 || q != p );
                q = q->cc_next;
            }
            assert( q == p );

            // check face
            u32 f = p->face_num;
            face_check |= 1uLL << f;
            q=p;
            assert( face_degree[p->face_num] > 0 );
            for (u32 i = 0; i < face_degree[p->face_num]; ++i ) {
                assert( q->face_num == p->face_num );
                assert( i==0 || q != p );
                q = q->other_end->cc_next;
            }
            assert( q == p );
            p = p->other_end;
        }
    }

    // check that all faces and nodes were seen
    assert( node_check == ( 1uLL << num_nodes ) - 1uLL );
    assert( face_check == ( 1uLL << num_faces ) - 1uLL );

    // check degrees
    u32 f = 0;;
    for (u32 ff = 0; ff < num_faces; ++f, ++ff ) {

        DEdge *e = face_edge[f];
        assert( face_degree[f] > 0 );
        assert( e->face_num == f );
        bool found = false;
        for ( DEdge *p = dedge_list; p; p = p->next ) {
            if ( p == e ) {
                found = true;
                break;
            }
        }
        assert ( found );
        DEdge *p = e;
        for ( u32 i = 0; i < face_degree[f]; ++i ) {
            assert( p != e || i == 0 );
            p = p->other_end->cc_next;
        }
        assert( p == e );
    }
    u32 n = 0;
    for (u32 nn = 0; nn < num_nodes; ++n, ++nn ) {
        DEdge *e = node_edge[n];
        assert( node_degree[n] > 0 );
        assert( e->node_num == n );
        bool found = false;
        for ( DEdge *p = dedge_list; p; p = p->next ) {
            if ( p == e ) {
                found = true;
                break;
            }
        }
        assert ( found );
        DEdge *p = e;
        for ( u32 i = 0; i < node_degree[n]; ++i ) {
            assert( p != e || i == 0 );
            p = p->cl_next;
        }
        assert( p == e );
    }

    // iterate through OEdge list and check degrees
    r = oedge_list;
    while(r) {
        u32 node_num = r->parent->node_num;
        OEdge *p = r;
        for ( u32 i =0; i < node_degree[node_num]; ++i ) {
            assert( i==0 || r != p );
            p = p->next_on_node;
        }
        assert( r == p );
        u32 face_num = *r->face_num_p;
        for ( u32 i =0; i < face_degree[face_num]; ++i ) {
            assert( i==0 || r != p );
            p = p->next_on_face;
        }
        assert( r == p );
        r = r->next;
    }
}
        
void Graph::create_from_string( const char* s ) 
{
    initialize();
    u32 node_num = 0;
    const char* p = s;
    DEdge *e = &edge_list->dedge_0;
    DEdge *node2edge[100];
    node2edge[0] = 0;
    node2edge[1] = e->other_end;

    // first char must connect node 0 to node 1;
    assert( *p == 'b' );
    ++p;

    for ( ; *p; ++p ) {
        if ( *p == ',' ) {
            ++node_num;
            e = node2edge[node_num]->cc_next;
            continue;
        }
        u32 new_node = *p - 'a';
        if ( new_node > node_num ) {
            // need to add an edge
            // e should be the insertion point.
            Edge *new_edge = Edge::allocate();
            if ( new_node == num_nodes ) {
                // first edge on this node
                node2edge[new_node] = &new_edge->dedge_1;
            }
            insert_edge( &new_edge->dedge_0, e, new_node );
            e = &new_edge->dedge_0;
        } else {
            // edge should already be there.
            // check that it is and move insertion point.
            e = e->cl_next;
            assert( e->other_end->node_num == new_node );
        }
    }
}

void Graph::debug_print()
{
    for ( Edge *e = edge_list; e; e = e->next ) {
        DEdge *d = &e->dedge_0;
        printf( "%d->%d   0x%x 0x%x    ", d->node_num, d->other_end->node_num, (u32)(u64)d, (u32)(u64)d->other_end );
        for ( DEdge* dd = d->cl_next; dd != d; dd = dd->cl_next ) {
            printf( "%d->%d   ", dd->node_num, dd->other_end->node_num );
        }
        printf("\n");

        /*
        printf( "        unreflected   next_on_node( 0x%x  0x%x )  next_on_face( 0x%x 0x%x )\n",
                  (u32)(u64)d->unreflected.next_on_node->parent,
                  (u32)(u64)d->other_end->unreflected.next_on_node->parent,
                  (u32)(u64)d->unreflected.next_on_face->parent,
                  (u32)(u64)d->other_end->unreflected.next_on_face->parent );
        printf( "          reflected   next_on_node( 0x%x  0x%x )  next_on_face( 0x%x 0x%x )\n",
                  (u32)(u64)d->reflected.next_on_node->parent,
                  (u32)(u64)d->other_end->reflected.next_on_node->parent,
                  (u32)(u64)d->reflected.next_on_face->parent,
                  (u32)(u64)d->other_end->reflected.next_on_face->parent );
                  */
    }
}


// Compare two OEdge objects determine if less than, equal, or greater than.
// returns -1 if ( *this < x )
// returns  0 if ( *this == x )
// returns  1 if ( *this > x )
// Comparison algorithm is arbitrary subject to the following conditions, and
// was picked for speed as this is the speed critical function for the entire
// grpah generation algorithm.
//
// returns x == y if and only if there is isomorphism from X to Y
//
// Obeys normal symmetry and transitivity of ordering operations.
//
// Is invariant across isomorphisms
//
int OEdge::compare( OEdge &x, bool first ) 
{
    // keep track of what nodes we have visited
    static u64 node_map1;
    static u64 node_map2;
    if ( first) {
        node_map1 = 0uLL;
        node_map2 = 0uLL;
    }
    node_map1 |= 1uLL << parent->node_num;
    node_map2 |= 1uLL << x.parent->node_num;


    Graph *g = grandparent->graph;
    u32 *node_degree = g->node_degree;
    u32 *face_degree = g->face_degree;
    u32  temp;

    temp =  node_degree[ parent->node_num ] - node_degree[ x.parent->node_num ];
    if ( temp ) return temp;

    temp =  node_degree[ other_end->parent->node_num ] - node_degree[ x.other_end->parent->node_num ];
    if ( temp ) return temp;

    temp =  face_degree[ *x.face_num_p ] - face_degree[ *face_num_p ];
    if ( temp ) return temp;

    temp =  face_degree[ *x.other_end->face_num_p ] - face_degree[ *other_end->face_num_p ];
    if ( temp ) return temp;

    // loop through all edges on this node
    OEdge *p = this;
    OEdge *q = &x;
    do {
        u32 other_end_node = p->other_end->parent->node_num;
        u32 other_end_node2 = q->other_end->parent->node_num;
        bool new_node1 = ( (node_map1 & 1uLL<<other_end_node) == 0uLL );
        bool new_node2 = ( (node_map2 & 1uLL<<other_end_node2) == 0uLL );
        if ( new_node1 && !new_node2)
            return 1;
        if ( !new_node1 && new_node2 )
            return -1;
        if ( new_node1 ) {
            // recurse only if other end is a new node.
            temp = p->next_on_face->compare( *q->next_on_face, false );
            if ( temp ) return temp;

            // once we have visited every node, then we are assured isomorphic
            // mapping if we haven't found discrepancy yet.
            if ( node_map1 + 1uLL == 1uLL << grandparent->graph->num_nodes )
                return 0;
        }

        p = p->next_on_node;
        q = q->next_on_node;
    } while ( p != this );

    return 0;
}
        

void OEdge::print_graph( char* output_string ) 
{
    const u32 MAX_NODE_COUNT = 32;
    assert( grandparent->graph->num_nodes <= MAX_NODE_COUNT );
    int node_name[MAX_NODE_COUNT];
    OEdge *node_first_edge[MAX_NODE_COUNT];
    memset( node_name, -1, sizeof(node_name) );
    u32 node_count = 1;
    u32 current_node = 0;
    node_name[ parent->node_num ] = 0;
    node_first_edge[0] = this;

    char* p = output_string;
    OEdge *e = this;

    while(1) {
        u32 other_end_node_num =  e->other_end->parent->node_num;
        if ( node_name[ other_end_node_num ] == -1 ) {
            node_first_edge[ node_count ] = e->other_end;
            node_name[ other_end_node_num ] = node_count++;
        }
        *p++ = (char)('a' + node_name[ other_end_node_num ] );

        // advance to next edge
        e = e->next_on_node;

        if ( e == node_first_edge[ current_node ] ) {
            e = node_first_edge[ ++current_node ];
            if ( current_node == grandparent->graph->num_nodes  ) {
                break;
            }
            *p++ = ',';
        }
    }
    *p = 0;
}

        
void Graph::remove_open( Edge* e )
{
    // remove face
    DEdge *d = &e->dedge_0;
    DEdge* doe = d->other_end;
    if ( d->face_num < doe->face_num ) {
        d = doe;
        doe = &e->dedge_0;
    }
    u32 face_to_remove = d->face_num;
    u32 remaining_face = doe->face_num;
    for ( DEdge *dd = d->other_end->cc_next; dd != d; dd = dd->other_end->cc_next ) {
        dd->face_num = remaining_face;
    }
    face_degree[ remaining_face ] += face_degree[ face_to_remove ] - 2;
    face_edge[ face_to_remove ] = 0;
    face_degree[ face_to_remove ] = 0;
    --node_degree[ e->dedge_0.node_num ];
    --node_degree[ e->dedge_1.node_num ];
    --num_faces;
    --num_edges;

    // check face_edge
    if ( face_edge[ remaining_face ] == doe )
        face_edge[ remaining_face ] = doe->other_end->cc_next;

    // disconnect the DEdges
    disconnect_edge( &e->dedge_0 );
    disconnect_edge( &e->dedge_1 );

    // remove edge from linked lists
    remove_edge_from_linked_lists( e );

    e->free();

    // if the face we removed was not the maximum face, then
    // we need to swap face number to make face numbers contiguous
    if ( face_to_remove != num_faces ) {
        DEdge* dd = face_edge[ num_faces ];
        do {
            dd->face_num = face_to_remove;
            dd = dd->other_end->cc_next;
        } while ( dd != face_edge[ num_faces ] );
        face_edge[ face_to_remove ] = face_edge[ num_faces ];
        face_degree[ face_to_remove ] = face_degree[ num_faces ];
        face_edge[ num_faces ] = 0;
        face_degree[ num_faces ] = 0;
    }
}

void Graph::remove_short( Edge* e )
{
    // remove node
    DEdge *d = &e->dedge_0;
    DEdge* doe = d->other_end;
    if ( d->node_num < doe->node_num ) {
        d = doe;
        doe = &e->dedge_0;
    }
    u32 node_to_remove = d->node_num;
    u32 remaining_node = d->other_end->node_num;
    for ( DEdge *dd = d->cc_next; dd != d; dd = dd->cc_next ) {
        dd->node_num = remaining_node;
    }
    node_degree[ remaining_node ] += node_degree[ node_to_remove ] - 2;
    node_edge[ node_to_remove ] = 0;
    node_degree[ node_to_remove ] = 0;
    --face_degree[ d->face_num ];
    --face_degree[ doe->face_num ];
    --num_nodes;
    --num_edges;

    // update face_edge if needed
    u32 f0_num = d->face_num;
    u32 f1_num = doe->face_num;
    if ( face_edge[f0_num] == d )
        face_edge[f0_num] = d->other_end->cc_next;
    if ( face_edge[f1_num] == doe )
        face_edge[f1_num] = doe->other_end->cc_next;

    DEdge *n1 = d->cc_next;
    DEdge *n2 = doe->cc_next;

    // disconnect the DEdges
    disconnect_edge( &e->dedge_0 );
    disconnect_edge( &e->dedge_1 );
    remove_edge_from_linked_lists( e );
    e->free();

    // now join the two nodes
    n1->cl_next->cc_next = n2;
    n2->cl_next->cc_next = n1;
    DEdge *temp = n1->cl_next;
    n1->cl_next = n2->cl_next;
    n2->cl_next = temp;

    // update the OE pointers
    n1->other_end->reflected.next_on_face = &n1->cl_next->reflected;
    n2->other_end->reflected.next_on_face = &n2->cl_next->reflected;
    n1->cl_next->other_end->unreflected.next_on_face = &n1->unreflected;
    n2->cl_next->other_end->unreflected.next_on_face = &n2->unreflected;

    n1->unreflected.next_on_node = &n1->cl_next->unreflected;
    n2->unreflected.next_on_node = &n2->cl_next->unreflected;
    n1->cl_next->reflected.next_on_node = &n1->reflected;
    n2->cl_next->reflected.next_on_node = &n2->reflected;

    // if the node we removed was not the maximum node, then
    // we need to swap node number to make node numbers contiguous
    if ( node_to_remove != num_nodes ) {
        DEdge* dd = node_edge[ num_nodes ];
        do {
            dd->node_num = node_to_remove;
            dd = dd->cl_next;
        } while ( dd != node_edge[ num_nodes ] );
        node_edge[ node_to_remove ] = node_edge[ num_nodes ];
        node_degree[ node_to_remove ] = node_degree[ num_nodes ];
        node_edge[ num_nodes ] = 0;
        node_degree[ num_nodes ] = 0;
    }
}

void Graph::remove_diamond( Edge* e ) 
{
    Edge *e_1 = e->dedge_0.cc_next->parent;
    Edge *e_2 = e->dedge_1.cl_next->parent;
    Edge *e_3 = e->dedge_0.cl_next->parent;
    Edge *e_4 = e->dedge_1.cc_next->parent;
    remove_open( e_1 );
    remove_open( e_4 );
    remove_short( e_2 );
    remove_short( e_3 );
}

void Graph::add_diamond( Edge* e ) 
{
    DEdge* d0 = &e->dedge_0;
    DEdge* d1 = &e->dedge_1;
    add_short( d0->cc_next, d0);
    add_short( d1->cc_next, d1);
    add_open( d1->cc_next, d0->cl_next->other_end );
    add_open( d0->cc_next, d1->cl_next->other_end );
}

// add an edge (reverse of open circuting an edge).
// ref0 and ref1 will be the first CC edges from the new edge
// the new edge will be cl_next for each of the ref edges.
void Graph::add_open( DEdge* ref0, DEdge* ref1 )
{
    // creates a new face, so allocate a new face
    u32 new_face = num_faces++;
    u32 old_face = ref0->face_num;
    while ( face_degree[ new_face ] > 0 ) {
       --new_face;
       assert( new_face < num_faces );
    } 

    // update face info
    u32 count=0;
    for ( DEdge* d = ref1; d != ref0; d = d->other_end->cc_next ) {
        ++count;
        d->face_num = new_face;
    }
    face_edge[old_face] = ref0;
    face_edge[new_face] = ref1;
    face_degree[old_face] -= count - 1;
    face_degree[new_face] = count + 1;

    // insert new edge
    Edge* e = Edge::allocate();
    connect_edge_cl( &e->dedge_0, ref0 );
    connect_edge_cl( &e->dedge_1, ref1 );
    ++num_edges;
    e->update_next( edge_list );
    edge_list = e;
    dedge_list = &e->dedge_0;
    oedge_list = &e->dedge_0.unreflected;
    e->graph = this;
    e->dedge_0.face_num = new_face;
    e->dedge_1.face_num = old_face;

    // update node info
    u32 n0 = ref0->node_num;
    u32 n1 = ref1->node_num;
    e->dedge_0.node_num = n0;
    e->dedge_1.node_num = n1;
    ++node_degree[n0];
    ++node_degree[n1];
}

void Graph::add_short( DEdge* ref0, DEdge* ref1 )
{
    // creates a new node, so allocate a new node
    u32 new_node = num_nodes++;
    u32 old_node = ref0->node_num;
    while ( node_degree[ new_node ] > 0 ) {
       --new_node;
       assert( new_node < num_nodes );
    } 

    // update node info
    u32 count=0;
    for ( DEdge* d = ref1; d != ref0; d = d->cc_next ) {
        ++count;
        d->node_num = new_node;
    }
    node_edge[old_node] = ref0;
    node_edge[new_node] = ref1;
    node_degree[old_node] -= count - 1;
    node_degree[new_node] = count + 1;

    // fix up node pointers
    ref0->cl_next->cc_next = ref1;
    ref1->cl_next->cc_next = ref0;
    DEdge* temp = ref0->cl_next;
    ref0->cl_next = ref1->cl_next;
    ref1->cl_next = temp;

    // insert new edge
    Edge* e = Edge::allocate();
    e->dedge_0.node_num = old_node;
    e->dedge_1.node_num = new_node;
    connect_edge_cl( &e->dedge_0, ref0 );
    connect_edge_cl( &e->dedge_1, ref1 );
    ++num_edges;
    e->update_next( edge_list );
    edge_list = e;
    dedge_list = &e->dedge_0;
    oedge_list = &e->dedge_0.unreflected;
    e->graph = this;
    e->dedge_0.face_num = ref1->face_num;
    e->dedge_1.face_num = ref0->face_num;

    ++face_degree[ref0->face_num];
    ++face_degree[ref1->face_num];
}

// this is a very slow routine for debug only.
// check that a graph is 3-connected
bool Graph::check_3_connected()
{
    for ( u32 i =0; i < num_nodes; ++i ) {
            for ( u32 j = 0; j < i; ++j ) {
                    // see if graph disconnects at nodes i and j
                    DEdge* stack_current[128];
                    DEdge* stack_limit[128];
                    int stack_pointer = 0;
                    DEdge *d = dedge_list;
                    // pick edge for starting point that is not either selected node
                    while ( d && ( d->node_num == i || d->node_num == j ) ) {
                        d = d->next;
                    }
                    assert( d );

                    stack_current[0] = d;
                    stack_limit[0] = d;
                    u64 nodes_marked = 1uLL << dedge_list->node_num  |  1uLL << i  |  1uLL << j;
                    while(stack_pointer >= 0) {
                        // do a depth first search stopping at any marked node.
                        d = stack_current[stack_pointer];
                        nodes_marked |= 1uLL << d->node_num;
                        if ( !( nodes_marked & 1uLL << d->other_end->node_num ) ) { 
                            ++stack_pointer;
                            stack_current[stack_pointer] = stack_limit[stack_pointer] = d->other_end;
                            continue;
                        }
                        stack_current[stack_pointer] = d->cl_next;
                        if ( stack_current[stack_pointer] == stack_limit[stack_pointer] ) {
                            // finished searching this node
                            --stack_pointer;
                        }
                    }
                    if ( nodes_marked != ( 1uLL << num_nodes ) - 1uLL ) {
                        return false;
                    }
            }
    }
    return true;
}

// this is a very slow routine for debug only.
// check that a graph is 2-connected
bool Graph::check_2_connected()
{
    for ( u32 i =0; i < num_nodes; ++i ) {
        // see if graph disconnects at node i
        DEdge* stack_current[128];
        DEdge* stack_limit[128];
        int stack_pointer = 0;
        DEdge *d = dedge_list;
        // pick edge for starting point that is not either selected node
        while ( d && ( d->node_num == i ) ) {
            d = d->next;
        }
        assert( d );

        stack_current[0] = d;
        stack_limit[0] = d;
        u64 nodes_marked = 1uLL << dedge_list->node_num  |  1uLL << i;
        while(stack_pointer >= 0) {
            // do a depth first search stopping at any marked node.
            d = stack_current[stack_pointer];
            nodes_marked |= 1uLL << d->node_num;
            if ( !( nodes_marked & 1uLL << d->other_end->node_num ) ) { 
                ++stack_pointer;
                stack_current[stack_pointer] = stack_limit[stack_pointer] = d->other_end;
                continue;
            }
            stack_current[stack_pointer] = d->cl_next;
            if ( stack_current[stack_pointer] == stack_limit[stack_pointer] ) {
                // finished searching this node
                --stack_pointer;
            }
        }
        if ( nodes_marked != ( 1uLL << num_nodes ) - 1uLL ) {
            return false;
        }
    }
    return true;
}

// debug routine to check that graph is simple and min degree 3.
void Graph::check_simple_mindeg_3()
{
    for ( DEdge* d = dedge_list; d; d = d->next ) {
        assert ( d->node_num != d->other_end->node_num );
        assert ( node_degree[ d->node_num ] >= 3 );
        for ( DEdge* dd = d->cl_next; dd != d; dd = dd->cl_next ) {
            assert ( dd->other_end->node_num != d->other_end->node_num );
        }
    }
}

// check if edge can be removed by opening without
// causing graph to become less than 3-connected.
// assumes graph is 3-connected before removal.
bool Graph::check_open_3con( Edge* e ) 
{
    u64 faces=0;
    // mark all faces connected to face adjacent to e.
    for ( DEdge* d = e->dedge_1.cc_next->other_end->cc_next; d != &e->dedge_0; d = d->other_end->cc_next ){
        for ( DEdge* dd = d->cl_next; dd!=d; dd = dd->cl_next ) {
            faces |= 1uLL << dd->face_num;
        }
    }
    // now check if any of the same faces are connected to the other face
    for ( DEdge* d = e->dedge_0.cc_next->other_end->cc_next; d != &e->dedge_1; d = d->other_end->cc_next ){
        for ( DEdge* dd = d->cl_next; dd!=d; dd = dd->cl_next ) {
            if ( faces & 1uLL << dd->face_num ) {
                return false;
            }
        }
    }
    return true;
}

// check if edge can be removed by opening without
// causing graph to become less than 2-connected.
// assumes graph is 2-connected before removal.
bool Graph::check_open_2con( Edge* e ) 
{
    u64 nodes=0;
    // mark nodes on face next to e.
    for ( DEdge* d = e->dedge_0.cc_next->other_end; d != &e->dedge_0; d = d->cc_next->other_end ) {
        nodes |= 1uLL << d->node_num;
    }
    for ( DEdge* d = e->dedge_1.cc_next->other_end; d != &e->dedge_1; d = d->cc_next->other_end ) {
        if ( nodes & 1uLL << d->node_num )
            return false;
    }
    return true;
}


// This is the dual of the above algorithm and checks if an edge can be 
// removed by shorting contingent on both adjacent faces having minimum size 4.
bool Graph::check_short_3con( Edge* e )
{
    u64 nodes=0;
    // mark all nodes on same face as node adjacent to e.
    for ( DEdge* d = e->dedge_0.cl_next; d->cl_next != &e->dedge_0; d = d->cl_next ){
        for ( DEdge* dd = d->other_end->cc_next; dd!=d; dd = dd->other_end->cc_next ) {
            nodes |= 1uLL << dd->node_num;
        }
    }
    // now check if any of the same nodes are connected by face to node at other end of e
    for ( DEdge* d = e->dedge_1.cl_next; d->cl_next != &e->dedge_1; d = d->cl_next ){
        for ( DEdge* dd = d->other_end->cc_next; dd!=d; dd = dd->other_end->cc_next ) {
            if ( nodes & 1uLL << dd->node_num ) 
                return false;
        }
    }
    return true;
}

// indicates if it is OK to remove a diamond
bool Graph::check_diamond_ok( Edge* e )
{
    return
       (    node_degree[ e->dedge_0.node_num ] == 3
         && node_degree[ e->dedge_1.node_num ] == 3
         && face_degree[ e->dedge_0.face_num ] == 3
         && face_degree[ e->dedge_1.face_num ] == 3 
         && node_degree[ e->dedge_0.cl_next->other_end->node_num ] > 3
         && node_degree[ e->dedge_0.cc_next->other_end->node_num ] > 3
         && ! e->dedge_0.cl_next->other_end->node_connects_to(  e->dedge_0.cc_next->other_end->node_num ) );
}

// this checks that shorting edge e will not cause graph
// to become 1-connected.  It also does additional check
// that shorting e will not cause any parallel edges.
bool Graph::check_short_2con( Edge* e ) 
{
    u64 faces = 0;
    u64 nodes = 0;
    for ( DEdge* d = e->dedge_0.cl_next; d != &e->dedge_0; d = d->cl_next ) {
        faces |= 1uLL << d->face_num;
        nodes |= 1uLL << d->other_end->node_num;
    }
    for ( DEdge* d = e->dedge_1.cl_next; d != &e->dedge_1; d = d->cl_next ) {
        if ( faces & 1uLL << d->face_num )
            return false;
        if ( nodes & 1uLL << d->other_end->node_num )
            return false;
    }
    return true;
}


bool Graph::is_wheel()
{
    // true if all nodes except one have degree 3
    // and all faces except one have degree 3.
    u32 count = 0;
    for ( u32 node = 0; node < 64; ++node ) {
        if ( node_degree[node] > 3 ) {
            ++count;
        }
    }
    if ( count > 1 )
        return false;
    count = 0;
    for ( u32 face = 0; face < 64; ++face ) {
        if ( face_degree[face] > 3 ) {
            ++count;
        }
    }
    if ( count > 1 )
        return false;
    return true;
}

void Graph::remove_edge_from_linked_lists( Edge *e )
{
    for ( Edge** p = &edge_list; *p; p = &(*p)->next ) {
        if ( *p == e ) {
            *p = e->next;
            break;
        }
    }
    for ( DEdge** p = &dedge_list; *p; p = &(*p)->next->next ) {
        if ( *p == &e->dedge_0 ) {
            *p = e->dedge_1.next;
            break;
        }
    }
    for ( OEdge** p = &oedge_list; *p; p = &(*p)->next->next->next->next ) {
        if ( *p == &e->dedge_0.unreflected ) {
            *p = e->dedge_1.reflected.next;
            break;
        }
    }
}



void Graph::create_wheel( u32 n ) 
{
    // creates wheel with 2*n edges, n+1 nodes and faces
    char string[200];
    char* p = string;
    for ( u32 i =0; i < n; ++i ) {
        *p++ = 'b' + (char)i;
    }
    for ( u32 i = 0; i < n; ++i ) {
        *p++ = ',';
        *p++ = 'a';
        *p++ = 'b' + ( i+ n - 1 ) % n;
        *p++ = 'b' + ( i+ n + 1 ) % n;
    }
    *p = 0;
    create_from_string( string );
}


StringSet::StringSet()
{
    size = 0;
}

void StringSet::reset()
{
    for ( u32 i=0; i < size; ++i ) {
        free_string( set[i] );
    }
    size = 0;
}

char* StringSet::allocate_string()
{
    char* string;
    if ( free_list  ) {
        string = (char*)free_list;
        free_list = *(char**)free_list;
    } else 
        string = (char*)malloc( 256 );
    return string;
}

void StringSet::free_string( char* string )
{
    *(char**)string = free_list;
    free_list = string;
}

bool StringSet::check_and_include( char * string ) 
{
    int low = 0;
    int high = size - 1;
    // do binary search to find if e is in set
    while ( high >= low ) {
        int mid = ( high + low ) / 2;
        int comp = strcmp( string, set[mid] );
        if ( comp == 0 ) {
            free_string( string );
            return true;
        }
        if ( comp < 0 ) {
            high = mid - 1;
        } else {
            low = mid + 1;
        }
    }
    // add e to set.  low points to insertion point
    for ( int i = (int)size; i > low; --i ) {
        set[i] = set[i-1];
    }
    set[low] = string;
    ++size;
    return false;
}

char* StringSet::free_list;

bool Graph::check_preferred_reduction( OEdge* e, ReductionType reduction_type, u32 connectivity ) 
{
    // checks if the given reduction is the preferred.
    // If tied with other reduction, then returns true, meaning
    // some other filter must be used to select a unique expansion.
    //
    // first check for a more preferred open reduction
    // assume that node numbers are contiguous (not the case for in general,
    // but should be for generation).
    u32 node_degree0;
    u32 node_degree1;
    bool c3 = ( connectivity == 3 );
    if ( reduction_type == OPEN ) {
        node_degree0 = node_degree[ e->parent->node_num ];
        node_degree1 = node_degree[ e->parent->other_end->node_num ];
    } else {
        // any valid open reduction is preferred to other reduction
        node_degree0 = 4;
        node_degree1 = 4;
    }
    if ( node_degree1 > node_degree0 )
        return false;
    for ( u32 n = 0; n < num_nodes;  ++n ) {
        u32 current_node_degree = node_degree[n];
        if ( current_node_degree < node_degree0 )
            continue;
        DEdge* dd = node_edge[n];
        DEdge* d = dd;
        do {
            u32 other_end_node_degree = node_degree[ d->other_end->node_num ];
            if ( other_end_node_degree < 4 )
                goto next;
            if ( current_node_degree == node_degree0 && other_end_node_degree < node_degree1 )
                goto next;
            // now do the slow checks
            if ( reduction_type == OPEN && d->unreflected <= *e && d->reflected <= *e )
                goto next;
            if ( !( c3 ? check_open_3con( d->parent ): check_open_2con( d->parent ) ) )
                goto next;
            // d is a better reduction.  Note that if e is a short or diamond reduction,
            // then any valid open reduction is preferred.
            return false;

            next:

            d = d->cl_next;
        } while ( d != dd );
    }

    if ( reduction_type == OPEN )
        return true;

    // check short reductions
    for ( u32 f = 0; f < num_faces;  ++f ) {
        DEdge* dd = face_edge[f];
        DEdge* d = dd;
        do {
            if ( face_degree[ d->face_num ] < 4 ) 
                goto next_short;
            if ( face_degree[ d->other_end->face_num ] < 4 ) 
                goto next_short;
            if ( reduction_type == SHORT && d->unreflected <= *e && d->reflected <= *e )
                goto next_short;
            if ( !( c3 ? check_short_3con( d->parent ) : check_short_2con( d->parent ) ) ) 
                goto next_short;
            return false;

            next_short:

            d = d->other_end->cc_next;
        } while ( d != dd );
    }

    if ( reduction_type == SHORT ) 
        return true;

    // at this point we must have a diamond reduction
    assert( !c3 );

    for ( Edge* x = edge_list; x; x = x->next ) {
        // check if e is a valid diamond reduction
        if ( !check_diamond_ok( x ) )
            continue;

        // at this point we determined that x is a valid diamond reduction.
        // Check if it is preferred to e
        if ( *max_oedge( x ) > *e )
            return false;
    }

    // no better reduction found
    return true;
}


OEdge* Graph::max_oedge( DEdge* d ) {
    return ( d->reflected > d->unreflected ) ? &d->reflected : &d->unreflected;
}

OEdge* Graph::max_oedge( Edge* e ) {
    OEdge* p0 = &e->dedge_0.unreflected;
    OEdge* p1 = &e->dedge_0.  reflected;
    OEdge* p2 = &e->dedge_1.unreflected;
    OEdge* p3 = &e->dedge_1.  reflected;
    if ( *p0 < *p1 )
        p0 = p1;
    if ( *p0 < *p2 )
        p0 = p2;
    if ( *p0 < *p3 )
        p0 = p3;
    return p0;
}

OEdge* Graph::max_oedge()
{
    // find the max OEdge in the entire graph.
    // Pre-screen based on node degrees.  This
    // speeds up the search a little.
    OEdge* emax = 0;
    u32 max_degree = 0;
    u32 max_degree_other_end = 0;
    for (DEdge* d = dedge_list; d; d=d->next ) {
        u32 n = node_degree[ d->node_num ];
        u32 noe = node_degree[ d->other_end->node_num ];
        if ( n > max_degree || ( n == max_degree && noe > max_degree_other_end ) ) {
            max_degree = n;
            max_degree_other_end = noe;
            emax = &d->unreflected;
        }
    }
    // use emax as starting point and search for max
    assert( emax );
    for ( OEdge* e = oedge_list; e; e = e->next ) {
        if ( e != emax && *e > *emax ) {
            emax = e;
        }
    }
    return emax;
}


// This is a simple hash code of the graph that should be invariant
// to isomorphism. Any two isomorphic graphs have the same hash code.
// Any two non-isomorphic graphs almost certainly have different hash codes.
// This is not used for generating graphs, but is very helpful for debugging.
u32 Graph::compute_hash()
{
    OEdge* e = max_oedge();
    u32 hash = 0xabcd1234;
    u64 nodemap = 0uLL;
    u32 sp = 0;
    OEdge* stack_current[64];
    OEdge* stack_begin[64];
    stack_current[sp] = e;
    stack_begin[sp] = e;
    // do a search of the graph
    while(1) {
        // first check other end of e
        u32 next_node = e->other_end->parent->node_num;
        if ( nodemap & 1uLL << next_node ) {
            // we have already seen this node
            hash = 3 * hash + 1;
            e = e->next_on_node;
            if ( e == stack_begin[sp] ) {
                // we have completed this node.  Pop back to previous node on stack.
                while ( 1 ) {
                    assert( sp > 0 );
                    --sp;
                    e = stack_current[sp]->next_on_node;
                    if ( e != stack_begin[sp] ) {
                        break;
                    }
                }
            }
        } else {
            // we have not seen this node yet.  Push current node on stack and search new node;
            nodemap |= 1uLL << next_node;
            hash = 13*hash + 5*node_degree[next_node] + face_degree[*e->face_num_p];
            if ( nodemap + 1 == 1uLL << num_nodes ) {
                // once we have seen all nodes, return the hash
                return hash;
            }
            stack_current[sp] = e;
            ++sp;
            e = e->other_end;
            stack_begin[sp] = e;
        }
    }
    return hash;
}


















 












