#include "Filter.h"
#include <stdio.h>
#include <assert.h>
#include <malloc.h>
#include <string.h>
#include <math.h>



void Filter::read_filter_data( char *file_name )
{
    FILE *infile = fopen( file_name, "r" );
    assert( infile );

    int allocated_size = 10000; // initial size, may get bigger
    data_array_size[0] = 0;
    data_array[0] = (DataArrayElement*)malloc( allocated_size * sizeof( DataArrayElement ) );
    memset( data_array_size, 0, sizeof( data_array_size ) );

    char *line_pointer = 0;
    size_t  line_size = 0;

    while ( getline( &line_pointer, &line_size, infile ) != -1 ) {
        if ( data_array_size[0] == allocated_size ) {
            allocated_size += allocated_size/2;
            data_array[0] = (DataArrayElement*)realloc( data_array[0], allocated_size * sizeof( DataArrayElement ) );
        }
        int order;
        int width;
        int height;
        int n = sscanf( line_pointer, "%d %d %d", &order, &width, &height );
        assert( n == 3 );
        DataArrayElement *e = &data_array[0][ data_array_size[0]++ ];
        e->order = order;
        assert( width >= height );
        e->ratio = (double)width / (double)height;
        if ( data_array_size[0] > 1 ) {
            assert( e->ratio > (e-1)->ratio );
        }
    }

    // get maximum order
    max_order = 0;
    for ( int i = 0; i < data_array_size[0]; ++i ) {
        int order = data_array[0][i].order;
        if ( max_order < order )
            max_order = order;
    }

    data_array[ max_order ] = data_array[0];
    data_array_size[ max_order ] = data_array_size[0];
    data_array[0] = 0;
    data_array_size[0] = 0;
    for ( int x = max_order-1; x >= 0; --x ) {
        // compute the data array for each order
        // first get the size
        for ( int y = 0; y < data_array_size[max_order]; ++y ) {
            if ( data_array[max_order][y].order <= x )
                ++data_array_size[x];
        }
        // allocate the array and fill it in
        data_array[x] = (DataArrayElement*)malloc( data_array_size[x]*sizeof( DataArrayElement ) );
        int p = 0;
        for ( int y = 0; y < data_array_size[max_order]; ++y ) {
            if ( data_array[max_order][y].order <= x )
                data_array[x][p++] = data_array[max_order][y];
        }
        assert ( p == data_array_size[x] );
    }
}




bool Filter::check_aspect_ratio( double aspect_ratio, int order )
{
    // aspect_ratio is compared to table entries with a tight tolerance, so it must be accurate.
    // In general matrix inversion is not sufficiently accurate, so this must be fixed by 
    // rounding the determinant to nearest integer, multiplying values as necessary by
    // determinant and rounding to create integer ratio , and finally dividing.

    if ( order >  max_order ) {
        return false;
    }

    double tolerance = 1e-13*aspect_ratio;
    int a = 0;
    int b = data_array_size[order] - 1;

    while ( b >= a ) {
        // do a binary search
        int c = ( a + b )/2;
        if ( fabs( aspect_ratio - data_array[order][c].ratio ) < tolerance )
            return true;
        if ( aspect_ratio > data_array[order][c].ratio ) {
            a = c+1;
        } else {
            b = c-1;
        }
    }
    return false;
}



