#include <stdio.h>
#include <assert.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>

#include "Boundary.h"


void Boundary::initialize( int width, int height ) {
    m_width = width;
    m_height = height;
    m_filled_area = 0;
    m_num_points = 1;
    points[0].x = 0;
    points[0].y = 0;
};

void Boundary::add_square( int left, int right, int bottom, int top ) {
    assert( left < right && bottom < top );
    assert( left >= 0 && bottom >= 0 );
    assert( right <= m_width && top <= m_height );
    m_filled_area += (top-bottom)*(right-left);
    // find point with x == left
    int a = -1;
    for ( int i = 0; i < m_num_points; ++i ) { 
        if ( points[i].x == left ) {
            a = i;
            break;
        }
    }
    assert( a >= 0 );
    assert( points[a].y == bottom );
    assert( a == m_num_points - 1 || points[a+1].x >= right );

    // figure out how the new square fits into the existing boundary
    bool bottom_consumes_segment =  ( right == m_width ) 
                                 || ( a < m_num_points - 1 && points[a+1].x == right );
    bool top_flush_left = ( a > 0 &&  points[a-1].y == top );
    bool top_flush_right = ( a < m_num_points - 1 && points[a+1].y == top );

    // handle each of the resulting cases
    if (  bottom_consumes_segment ) {
        if ( top_flush_left ) {
            if ( top_flush_right ) {
                // remve two segments, both a and a+1
                m_num_points -= 2;
                for ( int j = a; j < m_num_points; ++j ) {
                    points[j] = points[j+2];
                }
            } else { // !top_flush_right
                // remove segment a.
                --m_num_points;
                for ( int j = a; j < m_num_points; ++j ) {
                    points[j] = points[j+1];
                }
            }
        } else { // ! top_flush_left
            // move the current segment up
            points[a].y = top;
            if ( top_flush_right ) {
                // combine with segment to right
                --m_num_points;
                for ( int j = a+1; j < m_num_points; ++j ) {
                    points[j] = points[j+1];
                }
            }
        }
    } else { // !bottom_comsumes_segment
        if ( top_flush_left ) {
            // just extend segment to left
            points[a].x = right;
        } else { // !top_flush_left
            // need to create new segment
            for ( int j = m_num_points; j >= a; --j ) {
                points[j] = points[j-1];
            }
            ++m_num_points;
            points[a].y = top;
            points[a].x = left;
            points[a+1].x = right;
        }
    }
};

void Boundary::remove_square( int left, int right, int bottom, int top ) {
    m_filled_area -= (top-bottom)*(right-left);
    // reverses the operation of add_square
    // find rightmost pioint with x <= left
    int a = -1;
    for ( int i = 0; i < m_num_points; ++i ) { 
        if ( points[i].x <= left ) {
            a = i;
        } else {
            break;
        }
    }
    assert( a >= 0 );
    assert( points[a].y == top );
    assert( a == m_num_points - 1 || points[a+1].x >= right );

    // figure out how the square fits in
    bool bottom_consumes_segment =   a == m_num_points - 1
                                  ||   points[a+1].x != right
                                  ||   points[a+1].y != bottom;
    bool top_flush_left = ( points[a].x < left );
    bool top_flush_right =   ( a == m_num_points - 1 ) ? right < m_width
                                                       : right <  points[a+1].x;

    // handle each of the resulting cases
    if (  bottom_consumes_segment ) {
        if ( top_flush_left ) {
            if ( top_flush_right ) {
                // add two segments, both a and a+1
                for ( int j = m_num_points - 1; j >= a ; --j ) {
                    points[j+2] = points[j];
                }
                m_num_points += 2;
                points[a+1].x = left;
                points[a+1].y = bottom;
                points[a+2].x = right;
                points[a+2].y = top;
            } else { // !top_flush_right
                // add one segment
                for ( int j = m_num_points - 1; j >= a ; --j ) {
                    points[j+1] = points[j];
                }
                m_num_points += 1;
                points[a+1].x = left;
                points[a+1].y = bottom;
            }
        } else { // ! top_flush_left
            if ( top_flush_right ) {
                // add one segment
                for ( int j = m_num_points - 1; j >= a ; --j ) {
                    points[j+1] = points[j];
                }
                m_num_points += 1;
                points[a].y = bottom;
                points[a+1].x = right;
            }  else { // ! top_flush_right
                // just move current segment down
                points[a].y = bottom;
            }
        }
    } else { // !bottom_comsumes_segment
        if ( top_flush_left ) {
            //  move next segmeht back
            points[a+1].x = left;
        } else { // !top_flush_left
            // delete current segment - 
            --m_num_points;
            for ( int j = a; j < m_num_points; ++j ) {
                points[j] = points[j+1];
            }
            points[a].x = left;
        }
    }
};

    
void Boundary::get_loc_lowest( int &x, int&y ) {
    // returns x and y location of lowest unoccupied location (left most within lowest)
    x = points[0].x;
    y = points[0].y;
    for ( int i=1; i < m_num_points; ++i ) {
        int tempy = points[i].y;
        if ( tempy < y ) {
            x = points[i].x;
            y = tempy;
        }
    }
};

void Boundary::get_loc_well( int &x, int &y, int &width ) {
    // finds the narrowest well (horizontal segment with upward extending boundaries on both ends)
    width = m_width + 1;
    for ( int i=0; i < m_num_points; ++i ) {
        bool left_up = i==0 || points[i-1].y > points[i].y;
        bool right_up = i == m_num_points-1 || points[i+1].y > points[i].y;
        bool well = left_up && right_up;
        if ( !well )
            continue;
        int temp_width = (i == m_num_points-1 ) ? m_width - points[i].x  :  points[i+1].x - points[i].x;
        if ( temp_width  < width ) {
            width = temp_width;
            x = points[i].x;
            y = points[i].y;
        }
    }
    assert( width > 0 && width <= m_width );
};


void Boundary::check( ) {
    // just used for debugging
    int area_check = 0;
    assert( m_num_points > 0 );
    assert( points[0].x == 0 );
    for ( int i = 0; i < m_num_points; ++i ) {
        assert( i==0 || points[i].x > points[i-1].x );
        assert( i==0 || points[i].y != points[i-1].y );
        assert( points[i].x >= 0 && points[i].x < m_width );
        assert( points[i].y >= 0 && points[i].y <= m_height );
        int next_x = ( i < m_num_points - 1 ) ? points[i+1].x : m_width;
        area_check += points[i].y * ( next_x - points[i].x );
    }
    assert( area_check == m_filled_area );
}



