﻿// #########################################################################################################
// # simptc2canon - written by Lorenz Milla, Dossenheim, Germany, Version: 03/Nov/2013.                    #
// # reads in tablecode and converts it to canonical tablecode without looking at compound substructures.  #
// # requires one argument: the inputfilename.                                                             #
// # outputs one tablecode per line to inputfilename-simpcanon.txt,                                        #
// # where it adds a numbering in the end, counting through the different isomers.                         #
// # feel free to modify!                                                                                  #
// #########################################################################################################

#include <stdio.h>
#include <string.h>
#include <time.h>

const int MO = 100; //maximum order of the squared squares to process

int main(int argc, char *argv[]) {
  int tmp,tmpp;
  FILE *outfile;
  int i,j,k,x,z,ind,noncanon=0;
  char info[1024];
  char outfilename[256];
	
	int fl[512][MO],ft[512][MO],fr[512][MO],fb[512][MO]; // j-th element of the i-th isomer // max. 64*8 isomers!
  int l[MO],t[MO],r[MO],b[MO],tc[512][MO];
  int numsub;
  int L[100],T[100],R[100],B[100]; //maximum 100 subrectangles/subsquares ok!
  int n,w,h,numflips;
 
  int bigger(int a, int b) {
    //returns the index of the numerically bigger tablecode
    int ii,rr;
    rr=a;
    for(ii=n-1;ii>=0;ii--) {
      if(tc[b][ii]>tc[a][ii]) {rr=b; }
      if(tc[b][ii]<tc[a][ii]) {rr=a; }
    }
    return rr;
  }


  int allflips(int useflip, int sq, int pos) {
      // flips the sq-th dissection with the useflip-th subrectangle,
      // saves them to pos, pos+1, ... th position
       int zz=3;
       int ii,jj;
       if (R[useflip]-L[useflip]==B[useflip]-T[useflip]) {zz+=4;}

       for(ii=0;ii<n;ii++) { //create 4 or 8 isomers

         if((fl[sq][ii]>=L[useflip])&&(ft[sq][ii]>=T[useflip])&&(fr[sq][ii]<=R[useflip])&&(fb[sq][ii]<=B[useflip])) {

            fl[pos+0][ii]=L[useflip]+R[useflip]-fr[sq][ii]; //left-right-mirrored
            ft[pos+0][ii]=ft[sq][ii];
            fr[pos+0][ii]=L[useflip]+R[useflip]-fl[sq][ii];
            fb[pos+0][ii]=fb[sq][ii];

            fl[pos+1][ii]=fl[sq][ii];                         //top-bottom-mirrored
            ft[pos+1][ii]=T[useflip]+B[useflip]-fb[sq][ii];
            fr[pos+1][ii]=fr[sq][ii];
            fb[pos+1][ii]=T[useflip]+B[useflip]-ft[sq][ii];

            fl[pos+2][ii]=L[useflip]+R[useflip]-fr[sq][ii]; //lr&tb-mirrored
            ft[pos+2][ii]=T[useflip]+B[useflip]-fb[sq][ii];
            fr[pos+2][ii]=L[useflip]+R[useflip]-fl[sq][ii];
            fb[pos+2][ii]=T[useflip]+B[useflip]-ft[sq][ii];

            if (zz==7) {
              fl[pos+3][ii]=L[useflip]+(ft[sq][ii]-T[useflip]);   //original, 90 degrees rotated anticlockwise
              ft[pos+3][ii]=T[useflip]+(R[useflip]-fr[sq][ii]);
              fr[pos+3][ii]=R[useflip]-(B[useflip]-fb[sq][ii]);
              fb[pos+3][ii]=B[useflip]-(fl[sq][ii]-L[useflip]);

              fl[pos+4][ii]=L[useflip]+(ft[pos+0][ii]-T[useflip]);   //lr-mirrored, 90 degrees rotated anticlockwise
              ft[pos+4][ii]=T[useflip]+(R[useflip]-fr[pos+0][ii]);
              fr[pos+4][ii]=R[useflip]-(B[useflip]-fb[pos+0][ii]);
              fb[pos+4][ii]=B[useflip]-(fl[pos+0][ii]-L[useflip]);

              fl[pos+5][ii]=L[useflip]+(ft[pos+1][ii]-T[useflip]);   //tb-mirrored, 90 degrees rotated anticlockwise
              ft[pos+5][ii]=T[useflip]+(R[useflip]-fr[pos+1][ii]);
              fr[pos+5][ii]=R[useflip]-(B[useflip]-fb[pos+1][ii]);
              fb[pos+5][ii]=B[useflip]-(fl[pos+1][ii]-L[useflip]);

              fl[pos+6][ii]=L[useflip]+(ft[pos+2][ii]-T[useflip]);   //lr&tb-mirrored, 90 degrees rotated anticlockwise
              ft[pos+6][ii]=T[useflip]+(R[useflip]-fr[pos+2][ii]);
              fr[pos+6][ii]=R[useflip]-(B[useflip]-fb[pos+2][ii]);
              fb[pos+6][ii]=B[useflip]-(fl[pos+2][ii]-L[useflip]);

            }
            } else {
              for(jj=0;jj<zz;jj++) {
                fl[pos+jj][ii]=fl[sq][ii];
                ft[pos+jj][ii]=ft[sq][ii];
                fb[pos+jj][ii]=fb[sq][ii];
                fr[pos+jj][ii]=fr[sq][ii];
              }
            }
        }
        return zz;
  }


  int PosStep[100];
  int WidthStep[100];
  int HeightStep[100];
	int tc2ltrb(int n, int w, int h, int *code, int *l, int *t, int *r, int *b) { // returns 1 if ok, and 0 if error.
		int NumStep;
		int i,a,s,tmp,mergeleft,mergeright;
		
		// initialize: one empty step
		NumStep=1;
		PosStep[0]=0;
		WidthStep[0]=w;
		HeightStep[0]=0;
		i=0;
		do {
			s=0; for (a=0;a<NumStep;a++) { if (HeightStep[a]<HeightStep[s]) { s=a; } }
			l[i]=PosStep[s]; t[i]=HeightStep[s]; r[i]=l[i]+code[i]; b[i]=t[i]+code[i];
			tmp=code[i]-WidthStep[s];
			if (tmp>0) { return 0; } //square doesn't fit into gap!
			else if (tmp==0) { // code[i] fills up the whole step[s]
				if ((s>0)&&(HeightStep[s]+code[i]==HeightStep[s-1])) { mergeleft=1; } else { mergeleft=0; }
				if ((s<NumStep-1)&&(HeightStep[s]+code[i]==HeightStep[s+1])) { mergeright=1; } else { mergeright=0; }
				if ((mergeleft==1)&&(mergeright==1)) { // step[s-1],step[s],step[s+1] form one new step
					NumStep=NumStep-2;
					WidthStep[s-1] += WidthStep[s]+WidthStep[s+1];
					for(a=s;a<NumStep;a++) {
						PosStep[a]=PosStep[a+2];
						WidthStep[a]=WidthStep[a+2];
						HeightStep[a]=HeightStep[a+2];
					}
				}
				if ((mergeleft==1)&&(mergeright==0)) {// step[s-1],step[s] form one new step
					NumStep=NumStep-1;
					WidthStep[s-1] += WidthStep[s];
					for(a=s;a<NumStep;a++) {
						PosStep[a]=PosStep[a+1];
						WidthStep[a]=WidthStep[a+1];
						HeightStep[a]=HeightStep[a+1];
					}
				}
				if ((mergeleft==0)&&(mergeright==1)) {// step[s],step[s+1] form one new step
					NumStep=NumStep-1;
					WidthStep[s] += WidthStep[s+1];
					HeightStep[s] += code[i];
					for(a=s+1;a<NumStep;a++) {
						PosStep[a]=PosStep[a+1];
						WidthStep[a]=WidthStep[a+1];
						HeightStep[a]=HeightStep[a+1];
					}
				}
				if ((mergeleft==0)&&(mergeright==0)) {// step[s] is being filled up further
					HeightStep[s] += code[i];
				}
			} else if (tmp<0) { //step[s] is only partly filled, starting from left
				if ((s>0)&&(HeightStep[s]+code[i]==HeightStep[s-1])) { mergeleft=1; } else { mergeleft=0; }
				if (mergeleft==1) { //step[s-1] grows wider, step[s] gets thinner
					WidthStep[s-1] += code[i];
					WidthStep[s] -= code[i];
					PosStep[s] += code[i];
				}
				if (mergeleft==0) { //step[s] is being cut into two steps (one new)
					NumStep++;
					for(a=NumStep-1;a>=s+2;a--) {
						PosStep[a]=PosStep[a-1];
						WidthStep[a]=WidthStep[a-1];
						HeightStep[a]=HeightStep[a-1];
					}
					PosStep[s+1]=PosStep[s]+code[i];
					WidthStep[s+1]=WidthStep[s]-code[i];
					HeightStep[s+1]=HeightStep[s];
					WidthStep[s]=code[i];
					HeightStep[s]+=code[i];      
				}
			}
			i++;
		} while (i<n);
		if ((NumStep==1)&&(HeightStep[0]==h)&&(PosStep[0]==0)&&(WidthStep[0]==w)) { return 1; } else { return 0; }
		// if only one step is left, and this fills the whole rectangle, then the ltrb code is correct.
	}

  double tm = clock();
  if(argc==1) { printf("usage: simptc2canon.exe input.txt\n"); return 0; } else {
    FILE* file = fopen(argv[1], "r");
		if (file==NULL) { printf("failed to open %s\n",argv[1]); return 0; } else {
			printf("Processing tablecode from %s...\n",argv[1]);
			ind=0;

			sprintf(outfilename,"%s-simpcanon.txt",argv[1]);
			outfile=fopen(outfilename, "w");
			

			while(fscanf(file,"%d",&n) != EOF) {
				ind++;


				//read in tablecode
				fscanf(file,"%d",&w);            // w = width of rectangle
				fscanf(file,"%d",&h);            // h = height of rectangle
				for(i=0;i<n;i++) { fscanf(file,"%d",&tc[0][i]); }
        fgets(info, sizeof(info), file);
				size_t ln = strlen(info) - 1;
				if (info[ln] == '\n') info[ln] = '\0';



				// create ltrb code
				i=tc2ltrb(n, w, h, &tc[0][0], l, t, r, b);
				if(i==0) {printf("error in line %d\n",ind);}
				
				
				//for(k=0;k<n;k++) {printf("%d: %d %d %d %d\n",tc[0][k],l[k],t[k],r[k],b[k]);}

				numsub=0;
				L[numsub]=0; T[numsub]=0; R[numsub]=w; B[numsub]=h; // the whole square
     
     for(i=0;i<n;i++) { fl[0][i]=l[i]; ft[0][i]=t[i]; fr[0][i]=r[i]; fb[0][i]=b[i]; }
     numflips=1;
				
				
				 //flip the whole dissection
				 tmpp=0;
				 for(i=0;i<numflips;i++) { 
					 tmp=allflips(numsub,i,numflips+tmpp);
					 tmpp+=tmp; 
				 }
				 numflips += tmpp;
				
				 //first find all tablecodes
				 for(z=0;z<numflips;z++) {
					for(x=0;x<n;x++) {
						i=w*h*2;
						k=0;
						for(j=0;j<n;j++) {
							if (ft[z][j]*w+fl[z][j]<i) {
								i=ft[z][j]*w+fl[z][j];
								k=j;
							}
						}
						tc[z][x]=fb[z][k]-ft[z][k];
						ft[z][k]+=h; 
					}
				 }
				 //then find maximum tablecode (standard)
				 i=0;
				 for(j=1;j<numflips;j++) { //check j-th tablecode
					i=bigger(i,j);      
				 }
				 if(i>0){noncanon++;}
				 //the i-th isomer is the canonical now
				
			
				 fprintf(outfile,"%d %d %d",n,w,h);
				 for(j=0;j<n;j++) { 
					 fprintf(outfile," %d",tc[i][j]);
				 }
				 fprintf(outfile,"%s # %d.%d\n",info,ind,i+1);
			} // while(fscanf(file,"%d",&n) != EOF)
			if (feof(file)) {
				printf("End of file %s reached after %.3f seconds.\n",argv[1],(clock()-tm)/1000);
				printf("%d lines of tablecode analyzed.\n",ind);
				printf("They were converted to canonical and were saved to %s.\n",outfilename);
				printf("%d of those saved to %s haven't been canonical before.\n",noncanon,outfilename);
			} else { printf("End-of-File was not reached.\n"); }
			fclose(outfile);
			fclose(file);
    }
    return 0;
	}
}
