// author Stuart Errol Anderson;
// March 25 2024
// stuart.errol.anderson@gmail.com
//  bk2canon
//  bouwkampcode to canonical bouwkampcode
#include <vector>
#include <iostream>
#include <fstream>
#include <string>
#include <sstream>
#include <iomanip>
#include <utility>
#include <algorithm>
#include <numeric>
#include <iterator>
#include <climits>
#include <cmath>
#include <cstring>
#include <cstdlib>
#include <deque>
#include <exception>
#include <sys/time.h>
double get_time(){
	    struct timeval t;
	    gettimeofday(&t, NULL);
	    double d = t.tv_sec + (double) t.tv_usec/1000000;
	    return d;
	}
	
int main (int argc, char* argv[]){
	//create an ifstream infile

	double time_start = get_time();
	std::ifstream infile;
	//open ifstream infile
	infile.open(argv[1]);
	if(infile.fail()) {
		if (argc == 1) {
			std::cout<<" bk2canon (bouwkampcode to canonical) is an application"<<std::endl;
			std::cout<<" designed to read a squared rectangle or squared square bouwkampcode"<<std::endl;
			std::cout<<" and produce the canonical code of that tiling, in tablecode."<<std::endl;
			std::cout<<"  "<<std::endl;
			std::cout<<" Bouwkampcode is a code invented by C. J. Bowkamp."<<std::endl;
			std::cout<<" The purpose of bouwkampcode is to provide a concise notation for dissections"<<std::endl;
			std::cout<<" of rectangles (and squares) into squares. "<<std::endl;
			std::cout<<"  "<<std::endl;
			std::cout<<" Tablecode is bouwkampcode without parentheses and commas."<<std::endl;
			std::cout<<" The parenthesis around a group of numbers in bouwkampcode represent squares,"<<std::endl; 
			std::cout<<" with sizes given by the numbers, having their tops"<<std::endl;
			std::cout<<" at the same height and being contiguous from left to right."<<std::endl;
			std::cout<<"  "<<std::endl;
			std::cout<<" Tablecode consists of a string of numbers, firstly the number of squares in"<<std::endl;
			std::cout<<" the dissection, called the order, then the width and height of the dissection,"<<std::endl;
			std::cout<<" then the sizes of the squares, called the elements in order from the top left"<<std::endl;
			std::cout<<" to the right, filling the remaining gaps, from left to right until "<<std::endl;
			std::cout<<" the bottom of the dissection is reached and the dissection is tiled."<<std::endl;
			std::cout<<"  "<<std::endl;
			std::cout<<" A squared square or squared rectangle can be rotated and reflected in 8 ways"<<std::endl;
			std::cout<<" each of the 8 ways has a different bouwkampcode. One of those 8 codes is the"<<std::endl;
			std::cout<<" canonical code of that tiling.  For perfect squarings it is the code with "<<std::endl;
			std::cout<<" with the largest corner element in the top left square, and the larger "<<std::endl;
			std::cout<<" of the 2 elements adjacent to it on it's right.  For imperfect squarings the"<<std::endl;
			std::cout<<" same rule applies but further elements of the 8 codes may need to be compared,"<<std::endl;
			std::cout<<" element by element. The canonical code is the code with the highest "<<std::endl;
			std::cout<<" lexicographical value based on numerical comparison of corresponding elements "<<std::endl;
			std::cout<<" converted to a string. "<<std::endl;
			std::cout<<"  "<<std::endl;
			std::cout<<" Bouwkampcode can be converted to tablecode by replacing the commas and "<<std::endl;
			std::cout<<" parentheses with spaces. Tablecode can be converted to bouwkampcode by using"<<std::endl;
			std::cout<<" the tc2bkp program. tc2bkp and bk2canon and other programs are available from;"<<std::endl;
			std::cout<<" http://www.squaring.net/downloads"<<std::endl;
			std::cout<<" bk2canon usage; bk2canon filename"<<std::endl;
			std::cout<<" author Stuart Errol Anderson;"<<std::endl;
			std::cout<<" March 26, 2024"<<std::endl;
			std::cout<<" stuart.errol.anderson@gmail.com"<<std::endl;
		}
		else {
            std::cout<<" error; file "<<argv[1]<<" failed to load"<<std::endl;
	       	exit(0);
		}
		exit(0);
	}
	std::cout<<" Processing bouwkampcodes..."<<std::endl;
	//create a stringstream, strings for lines and values
	std::istringstream stream;
	std::istringstream ss;
	std::string line;
	std::string value;
	std::ofstream bkcanon_file;	// create ofstream  tiling bouwkampcode output file
   	std::ostringstream ous;
	ous <<"bkcanon-"<<argv[1];
	std::vector<int> Top ;
	std::vector<int> Bottom ;
	std::vector<int> Left;
	std::vector<int> Right ;
	std::vector<int> Size ;
	std::vector<int> Boundary ;
	std::vector<std::vector<int> > bkCodes;
	int counter = 0;
	int Order = 0;
	int BoundaryWidth = 0;
	int RectWidth = 0;
	int RectHeight = 0;	
	int element = 0;
   	int linecount = 0;

	while ( std::getline(infile, line, '\n' ) &&(line.length()>1)) { // extract lines
		 std::istringstream ins; // Declare an input string stream.
  		 size_t found;
  		 std::string linebk;
  		 std::string lineend;
  		 found=line.find("*");
  		 if (found != std::string::npos) {
              linebk = line.substr(0,found-1); // separate the bouwkampcode from the extended fields if they exist
              lineend = line.substr(found+1);
         } else {
        	  linebk = line;
        }
  		//if bouwkampcode convert to tablecode by deleting parentheses and commas and replacing with whitespace
		replace( linebk.begin(), linebk.end(), ',', ' ' );
		replace( linebk.begin(), linebk.end(), '(', ' ' );
		replace( linebk.begin(), linebk.end(), ')', ' ' );
		std::string newline;
  		newline += linebk;
  		newline += lineend;
  		ins.str(newline);        // Specify string to read.
		linecount ++;
		bool startrow = true;
		int s = 0;
		int startx = 0 ;
		int starty = 0 ;
		stream.clear();
		stream.str(newline); 
		while ( stream >> value  ) 	{ // get values in columns
			if (s <= 2) {
				if (s == 0) { 
					std::istringstream ss(value);
					ss >> Order ;//1st value is order
					ss.str("");
					int bksize = Order + 4;
		 			Top.reserve(bksize) ;
		 			Bottom.reserve(bksize) ;
		 			Left.reserve(bksize);
		 			Right.reserve(bksize) ;
		 			Size.reserve(bksize) ;
		 			Boundary.reserve(bksize) ;
		 			Size.push_back(Order);
		  			Top.push_back(0);
		  		    Left.push_back(0);
		  			Right.push_back(0);
	  				Bottom.push_back(0);
	  				Boundary.push_back(0);
				}
	  			if (s == 1) {
		  			std::istringstream	ss(value);
		  			ss >> RectWidth ;//2nd value is width
		  			BoundaryWidth = RectWidth;
		  			Size.push_back(RectWidth);
		  			Top.push_back(0);
		  		    Left.push_back(0);
		  			Right.push_back(0);
	  				Bottom.push_back(0);
	  				Boundary.push_back(0);
		  			ss.str("");
	  			}
				if (s == 2) {
					std::istringstream	ss(value);
					ss >> RectHeight ;//3rd value is height
	  				Size.push_back(RectHeight);
		  			Top.push_back(0);
		  		    Left.push_back(0);
		  			Right.push_back(0);
	  				Bottom.push_back(0);
	  				Boundary.push_back(0);
					ss.str("");
				} //
			} else {
				std::istringstream ss(value);
				 ss >> element;
				 ss.str("");
				Size.push_back(element);
			} //
			s++;
		} //
		// Place the squares ;  starting at top left, position successive squares from left to right in rows
		// the first row is the top edge of the tiling, successive rows descend in order of height.
		// If 2 (or more) rows have the same height, the leftmost one(s) come first.
		int l = 0; int n = 0;  
	    for (unsigned int j = 3; j <  Size.size(); j++){	
			if (startrow == true) {							
				Left.push_back(startx);		
				startrow = false;	
			} else {
			    
				Left.push_back(Left.at(j-1) + Size.at(j-1));
			}
			Right.push_back(Left[j] + Size[j]);
			Top.push_back(starty);
			Bottom.push_back(Top[j] + Size[j]);
			Boundary.push_back(j);
			for (unsigned int k = 0; k < j; k++) { // remove (make negative) old bottom square(s) from boundary list when covered by new one
			    l=Boundary[k];
				if (l>0){  
					if ((Top[j] == Bottom[l])&&(Left[j]<=Left[l])&&(Left[l]<=Right[j])) {
						BoundaryWidth = BoundaryWidth+Size[k];
						Boundary.at(k) = -Boundary[k];
					}
				}
			}
			if ((Right.at(j)-startx-BoundaryWidth) == 0){
				startrow = true;
				//v++;
				BoundaryWidth = 0;
				startx=INT_MAX; //start
  				starty=INT_MAX;
				for (unsigned int m=3; m <= j; m++){ // find minimum Boundary Bottom (starty) and Left (startx)
					n = Boundary[m];
					if ((n>0)&&(Bottom[n] < starty)){
						starty = Bottom[n];
						startx = Left[n];
					} else if ((n > 0)&&(Bottom[n] == starty)){
						if (startx > Left[n]){
							startx = Left[n];	
						}
					}
				}
			} else {
				startrow = false;	
			}
		} 
		//copy element position vectors
		std::vector<int> sortedTop(Top); //copy Top vector to sortedTop vector using constructor
		std::vector<int> sortedLeft(Left);// "      Left     "      "  sortedLeft  "          "           "
		std::vector<int> sortedRight(Right); //    copy Right vector to sortedRight vector using constructor
		std::vector<int> sortedBottom(Bottom);// "      Bottom "      "  sortedBottom  "          "           "		
		
		//eliminate 1st 3 values (zeroed order, width, height) from copied element position vectors
        sortedTop.erase (sortedTop.begin(),sortedTop.begin()+3);
        sortedLeft.erase (sortedLeft.begin(),sortedLeft.begin()+3);
        sortedRight.erase (sortedRight.begin(),sortedRight.begin()+3);
        sortedBottom.erase (sortedBottom.begin(),sortedBottom.begin()+3);
        
		//sort copied element position vectors
		std::sort(sortedTop.begin(), sortedTop.end()); // sort sortedTop vector in ascending order
		std::sort(sortedLeft.begin(), sortedLeft.end());//   "    sortedLeft   "       "      "               "
		std::sort(sortedRight.begin(), sortedRight.end()); //       sort sortedRight vector in ascending order
		std::sort(sortedBottom.begin(), sortedBottom.end());//   "    sortedBottom   "       "      "               "			
		
		//eliminate duplicate values to get unique sorted element position vectors
		sortedTop.erase( std::unique( sortedTop.begin(),sortedTop.end() ), sortedTop.end() ); // unique sortedTop vector in ascending order
		sortedLeft.erase( std::unique( sortedLeft.begin(),sortedLeft.end() ), sortedLeft.end() );//   "         sortedLeft   "       "      "               "			
		sortedRight.erase( std::unique( sortedRight.begin(),sortedRight.end() ), sortedRight.end() ); //                unique sortedTop vector in ascending order
		sortedBottom.erase( std::unique( sortedBottom.begin(),sortedBottom.end() ), sortedBottom.end() );//       "       sortedBottom   "   "      "               "		
							
		std::vector<int> compBkcode; 
		
		//1. create bouwkampcode Top Left -> Bottom Right; Top down
		compBkcode.push_back(Size.at(0));  // Order of squared square
		compBkcode.push_back(Size.at(1));  // Width  "      "             "
		compBkcode.push_back(Size.at(2));  // Height "     "             "
		for (unsigned int p=0; p < sortedTop.size(); p++){
			for (unsigned int q=0; q < sortedLeft.size(); q++){
				for (unsigned int r=3; r < Size.size(); r++){
					if((sortedTop.at(p)==Top.at(r))&&(sortedLeft.at(q)==Left.at(r))){
						compBkcode.push_back(Size.at(r));
						//std::cout<<Size.at(r)<<" ";
					}
				}
			}
		}
		//std::cout<<std::endl;
		bkCodes.push_back(compBkcode);
		compBkcode.clear();
 
		//2. create bouwkampcode Top Right -> Bottom Left;  Top down
		compBkcode.push_back(Size.at(0));  // Order of squared square
		compBkcode.push_back(Size.at(1));  // Width  "      "             "
		compBkcode.push_back(Size.at(2));  // Height "     "             "
		for (unsigned int p=0; p < sortedTop.size(); p++){
			for ( std::vector<int>::reverse_iterator qit =sortedRight.rbegin() ; qit < sortedRight.rend(); ++qit ) {
				for (unsigned int r=3; r < Size.size(); r++){
					if((sortedTop.at(p)==Top.at(r))&&(*qit==Right.at(r))){
						compBkcode.push_back(Size.at(r));
						//std::cout<<Size.at(r)<<" ";
					}
				}
			}
		}
		//std::cout<<std::endl;
		bkCodes.push_back(compBkcode);
		compBkcode.clear();
	
 		//3. create bouwkampcode Bottom Right -> Top Left ; Bottom up
		compBkcode.push_back(Size.at(0));  // Order of squared square
		compBkcode.push_back(Size.at(1));  // Width  "      "             "
		compBkcode.push_back(Size.at(2));  // Height "     "             "
		for ( std::vector<int>::reverse_iterator pit =sortedBottom.rbegin() ; pit < sortedBottom.rend(); ++pit ) {
			for ( std::vector<int>::reverse_iterator qit =sortedRight.rbegin() ; qit < sortedRight.rend(); ++qit ) {
				for (unsigned int r=3; r < Size.size(); r++){
					if((*pit==Bottom.at(r))&&(*qit==Right.at(r))){
						compBkcode.push_back(Size.at(r));
						//std::cout<<Size.at(r)<<" ";
					}
				}
			}
		}
		//std::cout<<std::endl;
		bkCodes.push_back(compBkcode);
		compBkcode.clear();
 
		//4. create bouwkampcode Bottom Left -> Top Right ; Bottom up
		compBkcode.push_back(Size.at(0));  // Order of squared square
		compBkcode.push_back(Size.at(1));  // Width  "      "             "
		compBkcode.push_back(Size.at(2));  // Height "     "             "
		for ( std::vector<int>::reverse_iterator pit =sortedBottom.rbegin() ; pit < sortedBottom.rend(); ++pit ) {
			for (unsigned int q=0; q < sortedLeft.size(); q++){
				for (unsigned int r=3; r < Size.size(); r++){
					if((*pit==Bottom.at(r))&&(sortedLeft.at(q)==Left.at(r))){
						compBkcode.push_back(Size.at(r));
						//std::cout<<Size.at(r)<<" ";
					}
				}
			}
		}
		//std::cout<<std::endl;
		bkCodes.push_back(compBkcode);
		compBkcode.clear();
 
 		//5. create bouwkampcode Top Left -> Bottom Right ; Left down
		compBkcode.push_back(Size.at(0));  // Order of squared square
		compBkcode.push_back(Size.at(2));  // Width  "      "             "
		compBkcode.push_back(Size.at(1));  // Height "     "             "
		for (unsigned int q=0; q < sortedLeft.size(); q++){
			for (unsigned int p=0; p < sortedTop.size(); p++){
				for (unsigned int r=3; r < Size.size(); r++){
					if((sortedTop.at(p)==Top.at(r))&&(sortedLeft.at(q)==Left.at(r))){
						compBkcode.push_back(Size.at(r));
						//std::cout<<Size.at(r)<<" ";
					}
				}
			}
		}
		//std::cout<<std::endl;
		bkCodes.push_back(compBkcode);
		compBkcode.clear();
 
 		//6. create bouwkampcode Top Right -> Bottom Left ; Right up
		compBkcode.push_back(Size.at(0));  // Order of squared square
		compBkcode.push_back(Size.at(2));  // Width  "      "             "
		compBkcode.push_back(Size.at(1));  // Height "     "             "
		for ( std::vector<int>::reverse_iterator qit =sortedRight.rbegin() ; qit < sortedRight.rend(); ++qit ) {
			for (unsigned int p=0; p < sortedTop.size(); p++){
				for (unsigned int r=3; r < Size.size(); r++){
					if((sortedTop.at(p)==Top.at(r))&&(*qit==Right.at(r))){
						compBkcode.push_back(Size.at(r));
						//std::cout<<Size.at(r)<<" ";
					}
				}
			}
		}
		//std::cout<<std::endl;
		bkCodes.push_back(compBkcode);
		compBkcode.clear();
 
 		//7. create bouwkampcode Bottom Right -> Top Left ; Right up
		compBkcode.push_back(Size.at(0));  // Order of squared square
		compBkcode.push_back(Size.at(2));  // Width  "      "             "
		compBkcode.push_back(Size.at(1));  // Height "     "             "
		for ( std::vector<int>::reverse_iterator qit =sortedRight.rbegin() ; qit < sortedRight.rend(); ++qit ) {
			for ( std::vector<int>::reverse_iterator pit =sortedBottom.rbegin() ; pit < sortedBottom.rend(); ++pit ) {
				for (unsigned int r=3; r < Size.size(); r++){
					if((*pit==Bottom.at(r))&&(*qit==Right.at(r))){
						compBkcode.push_back(Size.at(r));
						//std::cout<<Size.at(r)<<" ";
					}
				}
			}
		}
		//std::cout<<std::endl;
		bkCodes.push_back(compBkcode);
		compBkcode.clear();
 
 		//8. create bouwkampcode Bottom Left -> Top Right ; Left down
		compBkcode.push_back(Size.at(0));  // Order of squared square
		compBkcode.push_back(Size.at(2));  // Width  "      "             "
		compBkcode.push_back(Size.at(1));  // Height "     "             "
		for (unsigned int q=0; q < sortedLeft.size(); q++){
			for ( std::vector<int>::reverse_iterator pit =sortedBottom.rbegin() ; pit < sortedBottom.rend(); ++pit ) {
				for (unsigned int r=3; r < Size.size(); r++){
					if((*pit==Bottom.at(r))&&(sortedLeft.at(q)==Left.at(r))){
						compBkcode.push_back(Size.at(r));
						//std::cout<<Size.at(r)<<" ";
					}
				}
			}
		}
		//std::cout<<std::endl;
		bkCodes.push_back(compBkcode);
		compBkcode.clear();
		//canonical bouwkampcode string method
		//std::vector<unsigned int> compsum (8);
		std::stringstream ss;
		int wordsize;
		std::vector<std::string> bkrowstring;
		if ((RectWidth > RectHeight )||(RectWidth == RectHeight)) {
			wordsize = ceil(log10(RectWidth));
			if (ceil(log10(RectWidth))== log10(RectWidth)) wordsize++;
			} else {
			wordsize = ceil(log10(RectHeight));
			if (ceil(log10(RectHeight))== log10(RectHeight)) wordsize++;
		}
		
		//std::cout<<wordsize<<std::endl;
		unsigned int r = Size.size(); //r is bouwkampcode length
		unsigned int ssize = r*wordsize;
		for (unsigned int p = 0; p < 8 ; p++) {
			for (unsigned int q = 0; q < r ; q++) {
				ss << std::setw(wordsize) << std::setfill('0')<<bkCodes[p][q];
			}
			bkrowstring.push_back(ss.str());
			ss.str("");
			ss.clear();
		}

		//row qq with the largest string is the canonical row
		std::vector<std::string>::iterator result = std::max_element(bkrowstring.begin(), bkrowstring.end());
		int qq = std::distance(bkrowstring.begin(), result) ;
		//compsum.clear();
		bkrowstring.clear();
		
		//cout<<"write canonical code"<<endl;
		std::ostringstream bk;
		for (unsigned int j=0; j < Size.size() ; j++){
			bk<<bkCodes[qq][j]<<" ";
			//cout<<bkCodes[qq][j]<<" ";
		}
		//cout<<endl;
		bk<<std::endl;
		std::string line1 = "";
		bkcanon_file.open (ous.str().c_str(), std::ios::out|std::ios::app);
		if (!bkcanon_file) // check it opened ok
		{
				std::cerr << " Cannot open file " << ous.str() << " for output.\n"; 
		}
		line1 = bk.str();
		bk.clear();
		bk.str("");
		bkcanon_file << line1 ;
		std::cout << line1 ;
		counter++;
		line1.clear();
		line1 = "";				
		bkcanon_file.close();  
		ous.clear();
		bk.clear();
		bkCodes.clear();
		compBkcode.clear();
		Top.clear() ;
		Bottom.clear() ;
		Left.clear();
		Right.clear() ;
		Size.clear() ;
		Boundary.clear() ;
		sortedTop.clear() ;
		sortedBottom.clear() ;
		sortedLeft.clear();
		sortedRight.clear() ;		
	} 
	infile.close();  
	double time_end = get_time();
	std::cout.precision(5);
	if (linecount > 0) std::cout<<" "<<linecount <<" bouwkampcodes from file "<<argv[1]<<" processed."<<std::endl;
	std::cout<<" "<<counter<<" bouwkampcodes saved to bkcanon-"<<argv[1]<<std::endl;
	std::cout<<" This program has run for: "<< time_end - time_start<< " seconds."<<std::endl;
  	return 0;
}// end function
